/*
 * Copyright (c) 2009 Regents of the SIGNET lab, University of Padova and DOCOMO Communications Laboratories Europe GmbH.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Padova (SIGNET lab) nor the 
 *    names of its contributors may be used to endorse or promote products 
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED 
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
Ideal MAC: this class implement a very simple ideal MAC. It can simply receive and send one packet at the same time and it consider only
the trasmission delay. Each node in the coverage area of the source can correctly receive a packet after a delay computed on the basis of the packet size and the transmission rate in use. Intereference, collisions and packet loss probability are not taken into account
*/

#include "MAC_ideal.h"
#include <mobilenode.h>

//#define NC_DEBUG_MAC

/* \brief MACITimer timer is used to determine whene the MAC is busy due to transmissions or receptions
*/
void MACITimer::start(double time)
{
	Scheduler &s = Scheduler::instance();
	assert(busy_ == 0);

	busy_ = 1;
	stime = s.clock();
	rtime = time;
	assert(rtime >= 0.0);
	s.schedule(this, &intr, rtime);
	fflush(stdout);
}


void MACITimer::stop(void)
{
	Scheduler &s = Scheduler::instance();
	assert(busy_);
	s.cancel(&intr);

	busy_ = 0;
	stime = 0.0;
	rtime = 0.0;
}

void MACITimer::handle(Event *)
{       
	busy_ = 0;
	stime = 0.0;
	rtime = 0.0;
	mac->txHandle();
}

void MACITimeout::handle(Event *)
{       
	busy_ = 0;
	stime = 0.0;
	rtime = 0.0;
	mac->timeout();
}


static class MacIClass : public TclClass {
public:
	MacIClass() : TclClass("Mac/MacI") {}
        TclObject *create (int, const char *const *) {
                return (new MacI ());
        }
} class_macI;



MacI::MacI() : Mac(), txTimer_(this), toutTimer_(this)
{
	node_ = 0;
	txState_ = 0;
	txPkt = 0;
	node_ = 0;
	next_ = 0;
	header_ = 0;
	size_ = 86;

	bind("header_", &header_);
	bind("timeout_", &timeout_);
	bind("bandwidth_", &bandwidth_);
	bind_bool("drop_on_timeout_", &drop_on_timeout_);

	bind("send_count", &send_count);
	bind("flooding", &flooding);
	bind("range_", &range_);
	
	mac_send_pkt = 0;
	mac_recv_pkt = 0;
	mac_recv1_pkt = 0;
	mac_col_pkt = 0;
	time_pkt = Scheduler::instance().clock();
	
}


int MacI::initialized()
{
	return (node_ != 0 && downtarget_ != 0 && uptarget_ != 0);
}

int MacI::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	if(argc == 2) {   
    		if(strncasecmp(argv[1], "id", 2) == 0) {
      			tcl.resultf("%d", index);
      			return TCL_OK;
    		}
	}

	if (argc == 3) 
	{
		if (strcmp(argv[1], "node") == 0) 
		{
			node_ = (MobileNode*) TclObject::lookup(argv[2]);
			if (node_ == 0)
				return TCL_ERROR;
			return TCL_OK;
		}
	}
	if (strcmp(argv[1],"print_stat") == 0)  {

		if (flooding) {
			sprintf(filename,"./Results/MAC_F_%i_%.2f.txt", node_->address(), send_count);
			fileout = fopen(filename,"a+");
			fprintf(fileout,"%i %f %i %i %i %i %i %i \n", node_->address(), time_pkt, mac_send_pkt, mac_recv_pkt, 0, mac_recv1_pkt, 0, 0);
			fclose(fileout);
		} else {
			sprintf(filename,"./Results/MAC_NC_%i_%.2f.txt", node_->address(), send_count);
			fileout = fopen(filename,"a+");
			fprintf(fileout,"%i %f %i %i %i %i %i %i \n", node_->address(), time_pkt, mac_send_pkt, mac_recv_pkt, 0, mac_recv1_pkt, 0, 0);
			fclose(fileout);
		}
		
      		return TCL_OK;
	}
	return Mac::command(argc, argv);
}


/* MacI::recv() function is used to receive packets both from upper and lower layer. If the direction of the packet is down, the packet is sent to 	   the send function, if it is up, the packet is processed and passed at the upper layers
*/

void MacI::recv(Packet *p, Handler *h)
{
	struct hdr_cmn *hdr = HDR_CMN(p);
	struct hdr_ip *ih = HDR_IP(p);
	/*
	 * Sanity Check
	 */
	assert(initialized());

	/*
	 *  Handle outgoing packets.
	 */
	if(hdr->direction() == hdr_cmn::DOWN) {
                send(p, h);
                return;
        }
	
	++mac_recv1_pkt;
	
	MobileNode* near_ = (MobileNode*)node_->get_node_by_address(ih->saddr());
	double dist_ = node_->distance(near_);
	
	if((ih->daddr() == node_->address() || ih->daddr() == IP_BROADCAST) && (ih->saddr() != node_->address()) && (dist_< range_))
	{
#ifdef NC_DEBUG_MAC
 		printf("\n%f - MacI - node %i - RICEVUTO UN PACCHETTO (%i)\n\n", Scheduler::instance().clock(), node_->address(), hdr->uid());
#endif
		++mac_recv_pkt;
		time_pkt = Scheduler::instance().clock();
		uptarget_->recv(p->copy(), (Handler*) 0);
	}
	Packet::free(p);
}


/* This function is used to initialize the packet Mac header, compute the transmission time and send the packet. 
*/
void MacI::send(Packet *p, Handler *h)
{
#ifdef NC_DEBUG_MAC
	printf("(%f) MACI send: Node %i \n", Scheduler::instance().clock(), node_->address());
#endif
	struct hdr_cmn *ch = HDR_CMN(p);
	
	double duration = 0;
	double rate = 0;
	
	txPkt = p;
	ch->size() = size_;
	callback_ = h;
	txState_ = 1;
#ifdef NC_DEBUG_MAC
 	printf("MacI (nodo %i) - arrivato pacchetto da inviare (uid=%i)\n", node_->address(), ch->uid());
#endif	
	rate = bandwidth_;
	
	if(rate > 0)
	{
		/*
		The rate is valid, thus I send the packet normally
		*/
		duration = (header_*8.0)/1000000 + (((double)ch->size())*8.0/rate);
		txTimer_.start(duration);
	}
	else
	{
		/*
		The rate is not valid, thus I launch the timeout timer
		*/
		toutTimer_.start(timeout_);
	}
}


/*
TIMER HANDLER
*/
void MacI::txHandle()
{
#ifdef NC_DEBUG_MAC
	printf("MACI txHandle \n");
#endif
	Handler *h = callback_;
	callback_ = 0;
	if(txPkt)
		transmit();
	Packet::free(txPkt);
	txPkt = 0;
	txState_ = 0;
	
	if (h)
		h->handle((Event *)0);
}

void MacI::timeout()
{
#ifdef NC_DEBUG_MAC
	printf("MACI timeout \n");
#endif
	double rate = 0;
	assert(txPkt);
	
	struct hdr_cmn *ch = HDR_CMN(txPkt);
	rate = bandwidth_;

	if(rate > 0)
	{
		/*
		The rate is valid, thus I send the packet normally
		*/
		double duration = header_/1000000 + (((double)ch->size())*8.0/rate);
		txTimer_.start(duration);
	}
	else if(drop_on_timeout_)
	{
		Handler *h = callback_;
		callback_ = 0;
		drop(txPkt, "OUT");
		txPkt = 0;
		txState_ = 0;
		
		if (h)
			h->handle((Event *)0);
	}
	else
	{
		toutTimer_.start(timeout_);
	}
	
}

// function passes the packet to the channel. 

void MacI::transmit()
{
#ifdef NC_DEBUG_MAC
	printf("MACI transmit: Node %i \n", node_->address());
#endif
	++mac_send_pkt;
	downtarget_->recv(txPkt->copy(), this);
}

void MacI::unblock()
{
#ifdef NC_DEBUG_MAC
	printf("macI (nodo %i)- unblock\n", node_->address());
#endif
	if(txState_ == 1 && toutTimer_.busy())
	{
		toutTimer_.stop();
		timeout();
	}
}
