/*
 * Copyright (c) 2009 Regents of the SIGNET lab, University of Padova and DOCOMO Communications Laboratories Europe GmbH.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Padova (SIGNET lab) nor the 
 *    names of its contributors may be used to endorse or promote products 
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED 
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef IMAC_
#define IMAC_

#include "address.h"
#include "ip.h"

#include "mac-timers.h"
#include "marshall.h"

#include <mobilenode.h>

/**
This class implement a very simple ideal MAC. It can simply receive and send one packet at the same time and it consider only
the trasmission delay. Each node in the coverage area of the source can correctly receive a packet after a delay computed on the basis of the packet size and the transmission rate in use. Intereference, collisions and packet loss probability are not taken into account
*/

class MacI;

/** This timer is used to determine whene the MAC is busy due to transmissions or receptions
*/
class MACITimer: public Handler
{
	public:
		MACITimer(MacI* m) : mac(m), intr() {
			busy_ = 0; stime = rtime = 0.0;
		}
	
		virtual void handle(Event *e);
	
		virtual void start(double time);
		virtual void stop(void);
	
		inline int busy(void) { return busy_; } 
		inline double expire(void) {
			return ((stime + rtime) - Scheduler::instance().clock());
		}
	
	protected:
		MacI		*mac;
		int		busy_;
		Event		intr;
		double		stime;	// start time
		double		rtime;	// remaining time
};

class MACITimeout: public MACITimer
{
	public:
		MACITimeout(MacI* m) : MACITimer(m) {}
	
		virtual void handle(Event *e);
};

class MacI : public Mac
{
	friend class MACITimer;
	friend class MACITimeout;
	protected:
		/** This function is used to initialize the packet Mac header, compute the transmission time and send the packet. */
		virtual void send(Packet *p, Handler *h);
		/** function passes the packet to the channel.*/ 
		virtual void transmit();
		virtual void txHandle();
		virtual void timeout();
		virtual int initialized();
		MACITimer	txTimer_;
		MACITimeout	toutTimer_;
		
		nsaddr_t        index_;	// IP Address of this node

		int		txState_;
		
		MobileNode* 	node_;
		
		Packet 		*txPkt;
		
		MacI		*next_;
		
		int		header_;
		
		double		timeout_;
		int 		drop_on_timeout_;
		/// \brief Transmission rate
		double		bandwidth_;

		FILE* fileout;
		char filename[70];
		
		int mac_send_pkt;
		int mac_recv_pkt;
		int mac_recv1_pkt;
		int mac_col_pkt;
		double time_pkt;
		double send_count;
		int flooding;
		int range_;
		int size_;
	
	public:
		MacI();
		/** It is used to receive packets both from upper and lower layer. If the direction of the packet is down, the packet is sent to 	   * the send function, if it is up, the packet is processed and passed at the upper layers
		*/
		virtual void recv(Packet* p, Handler* h);
		virtual int command(int argc, const char*const* argv);
		inline void setNext(MacI *mac) {next_ = mac;}
		inline MacI *next() {return next_;}
		
		inline MobileNode *getNode() {return node_;}
		
		virtual void unblock();
};

#endif
