/*
 * Copyright (c) 2009 Regents of the SIGNET lab, University of Padova and DOCOMO Communications Laboratories Europe GmbH.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Padova (SIGNET lab) nor the 
 *    names of its contributors may be used to endorse or promote products 
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED 
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <sys/time.h>
#include <limits.h>
#include <assert.h>
#include <sys/types.h>
#include <unistd.h>

#include <ip.h>
//#include <random.h>
#include <cmu-trace.h>
#include <energy-model.h>
#include <ll.h>

#include "NCR.h"
#include "rng_NC.h"
#include "nc_buffer.h"

#define max(a,b)        ( (a) > (b) ? (a) : (b) )
#define CURRENT_TIME    Scheduler::instance().clock()

//#define DEBUG
//#define ERROR

//#define NC_DEBUG
//#define NC_DEBUG_final
//HACK
//#define NC_DEBUG_fredi


#define DBL_MAX 2.2250738585072014e-308
#define LISTEN_ENERGY (0.0132 + 0.0165 * 0.03)
#define TX_ENERGY 0.38272
#define RX_ENERGY (0.30015 * 1.5)


#ifdef USE_NTL
#include <NTL/vec_GF2E.h>
using namespace NTL;
#else
#include <gf2e.h>
#endif

using namespace std;

// Unique counter for the generated packets
static int PktID = -1;

// Global variables
static int seed = 13113;
static NCBufferAllocation global_allocation;
static long max_packets = 2*NCBuffer::block_size*100;


//
// NCR header
//

int hdr_NC::offset_;

static class NCHeaderClass : public PacketHeaderClass {
public:
        NCHeaderClass() : PacketHeaderClass("PacketHeader/NC",
                                              sizeof(hdr_NC)) {
	bind_offset(&hdr_NC::offset_);
	}
	void export_offsets() {
		field_offset("len_", OFFSET(hdr_NC, len_));
		field_offset("gen_", OFFSET(hdr_NC, gen_));
		field_offset("coefficient_", OFFSET(hdr_NC, coefficient_));
		field_offset("lenCoeff_", OFFSET(hdr_NC, lenCoeff_));
		field_offset("time_", OFFSET(hdr_NC, time_));
	} 
} class_NChdr;

static class NCRclass : public TclClass {
public:
        NCRclass() : TclClass("Agent/NCR") {}
        	TclObject* create(int argc, const char*const* argv) {
          		assert(argc == 5);
          		return (new NCR((nsaddr_t) Address::instance().str2addr(argv[4])));
        }
} class_rtProtoNCRT;



//
// Protocol TIMER handler
//

NCR_timer::NCR_timer(NCR* agent) : TimerHandler() 
{
	agent_ = agent;
	interval_ = 0;
	active_ = 0;
	stop_time = false;
	agent_->bind("interval_", &interval_);
#ifdef NC_DEBUG_final
	printf("Timer: active_ = %i, interval_ = %f \n", active_, interval_);
#endif
}

void NCR_timer::expire(Event *e)
{	
	active_ = 0;
	if(!stopping_s){
		for(int ri = 0; ri< agent_->get_recv_inn(); ri++)
			agent_->sendCombination(0.0);
		agent_->set_recv_inn(0);
	} else if(stopping_s && !stop_time){
		if(agent_->NC_rng->genrand_real() <= agent_->get_sendcount())
			agent_->sendCombination(0.0);
		double delay = 2*agent_->NC_rng->genrand_real()*get_interval()*agent_->get_tau();
		start(delay,1);
	}
}

void NCR_timer::start(double delay, int s)
{
	active_ = 1;
	stopping_s = s;
	resched(delay);
}


int
NCR::command(int argc, const char*const* argv) {
	Tcl& tcl = Tcl::instance();
 
	if(argc == 2) {   
    		if(strncasecmp(argv[1], "id", 2) == 0) {
      			tcl.resultf("%d", index);
      			return TCL_OK;
    		}
    
    		if(strncasecmp(argv[1], "start", 2) == 0) {
			return (TCL_OK);
     		}
	} else if (strcmp(argv[1], "get_rank") == 0) {
			long int g_tmp = atol(argv[2]);
			long int rango = get_rank_of_g(g_tmp);
			if (rango < 0)
				return (TCL_ERROR);
			tcl.resultf("%li", rango);
			return (TCL_OK);
	} else if (argc == 3) {
		TclObject *obj;
                if ((obj = TclObject::lookup(argv[2])) == 0) {
                        fprintf (stderr, "%s: %s lookup of %s failed\n", __FILE__, argv[1],
                                 argv[2]);
                        return TCL_ERROR;
                }
                if (strcasecmp (argv[1], "tracetarget") == 0) {
                        tracetarget = (Trace *) obj;
                        return TCL_OK;
                }
                else if (strcasecmp (argv[1], "node") == 0) {
			
                        nodo = (MobileNode*)TclObject::lookup(argv[2]);
			if (nodo == 0 ) return TCL_ERROR;
                        return TCL_OK;
                }
                else if (strcasecmp (argv[1], "port-dmux") == 0) {
                        port_dmux_ = (NsObject *) obj;
                        return TCL_OK;
                }
		else if (strcmp(argv[1], "target") == 0) {
			if (*argv[2] == '0') {
				target_ = 0;
				return (TCL_OK);
			}
			target_ = (NsObject*)TclObject::lookup(argv[2]);
			if (target_ == 0) {
				tcl.resultf("no such object %s", argv[2]);
				return (TCL_ERROR);
			}
			
			return (TCL_OK);
		}		
		else if (strcmp(argv[1], "drop-target") == 0) {
			drop_ = (NsObject*)TclObject::lookup(argv[2]);
			if (drop_ == 0) {
				tcl.resultf("no object %s", argv[2]);
				return (TCL_ERROR);
			}
			return (TCL_OK);
		} 
	}

	if(strcmp(argv[1],"update_n") == 0) {
			bool ok = update_neigh();
			if (!ok)
				return (TCL_ERROR);
			return (TCL_OK);
     	}
	// This function prints all the statistical results
	if (strcmp(argv[1],"print_stat") == 0)  {
		
		
		fileout = fopen(filename,"a+");
		
                if (flooding) {
			fprintf(fileout,"%i %f %i %i %f %f %i %i %i %i \n", index_, statistic->time_inn, 0, statistic->app_decd, statistic->app_delay, statistic->app_dcdelay/statistic->app_decd, statistic->app_send, statistic->app_recv, 0, 0);
			fclose(fileout);
		} else {
			fprintf(fileout,"%i %f %f %i %f %f %i %i %i %i\n", index_, statistic->time_inn, statistic->app_delay, statistic->app_decd, statistic->sum_dcdelay, statistic->app_dcdelay/statistic->app_decd, statistic->app_send, statistic->app_recv, statistic->route_send, statistic->hello_send);
			fclose(fileout);

			file_inn = fopen(filename_inn,"a+");
			fprintf(file_inn,"%i %f %i %i %i %i %i %i \n", index_, statistic->time_inn, statistic->app_inn, statistic->app_recv, num_vicini, statistic->route_send, N, buffer[0]->get_notify());
			fclose(file_inn);

			file_topo = fopen(filename_topo,"a+");
			fprintf(file_topo,"%i %f %f \n", index_, nodo->X(), nodo->Y());
			fclose(file_topo);	
		}
		
      		return TCL_OK;
	}
	// -------------------------------
  	return Agent::command(argc, argv);
}

// ************* 
// Constructor
// *************

NCR::NCR (nsaddr_t id): Agent (PT_NC), port_dmux_(0), dmux_(0), use_mac_(0), nodo(0), timer_(this)
{

#ifdef NC_DEBUG
	printf("NCR:NCR() \n"); 
#endif
	index_ = id;
	sendPkt_ = 0;
	datasize_ = 0;
	neighbor = 0;
	recv_inn = 0;
	tau_avg=0.0;
	for (int i = 0; i < 100; i++)
		vicini[i] = -1;
	// Buffer inizialization: 
	for (int b = 0; b < MAX_BUFFERS; ++b){
		buffer[b] = 0;
		buf_checked[b] = 0;
	}

	num_buffers = 0;
	tx_buffer = -1;
	statistic = new NodeStats();
  	num_vicini = 0;
	int count_hello = 0;

	// For flooding
	seqno_idx = 0;
	seqno_low = 0;
	for (int j = 0; j < SEQNO_CACHESIZE; ++j)
		seqno_cache[j] = 0;

	// Binding variables with the Otcl. This variables can be set up from the Tcl script

	bind("sendPkt_", &sendPkt_);        	
	bind("channel_coding", &c_c);
	bind("probabilistic_nc", &p_n);
	bind("force_first", &f_f);
	bind("statistics", &st);
	bind("ncbs_hop_count", &ncbs_hop_count);
	bind("generation_management", &generation_management);
	bind("send_count", &send_count);
	bind("adaptive_send_count", &adaptive_send_count);
	bind("flooding", &flooding);
	bind("pseudo", &pseudo);
	bind("scenario", &scenario);
	bind("N", &N);
	bind("R", &R);
	bind("simple_strategy", &simple_strategy);
	bind("wanted_strategy", &wanted_strategy);	
	bind("stopping_strategy", &stopping_strategy);
	bind("full_matrix", &full_matrix);
	bind("full_decoded", &full_decoded);

	bind("num_hello", &num_hello);
	bind("generation_size", &generation_size);

	channel_coding = (c_c == 1);
	probabilistic_nc = (p_n == 1);
	force_first = (f_f == 1);
	statistics = (st == 1);	
	
	// Inizialization of the statistics files

	if (flooding) {
		sprintf(filename,"./Results/NCR_F_%i_%.2f.txt", index_, send_count);
		sprintf(filename_inn,"./Results/INN_F_%.2f.txt", send_count);
	} else {
		sprintf(filename,"./Results/NCR_NC_%i_%.2f.txt", index_, send_count);
		sprintf(filename_inn,"./Results/INN_NC_%.2f.txt", send_count);
		sprintf(filename_topo,"./Results/TOPO_NC.txt");
		//sprintf(filename_delay,"./Results/DELAY_NC.txt");
	}

	// Inizialization of the different modules
	init(); 
}

// *************
// Functions for the link layer failures
// *************

static void
ncr_rt_failed_callback(Packet *p, void *arg) 
{
#ifdef NC_DEBUG
	printf("NCR:cr_rt_failed_callback() \n"); 
#endif
	((NCR*) arg)->rt_ll_failed(p);
return;
}

void
NCR::rt_ll_failed(Packet *p) 
{
#ifdef NC_DEBUG
	printf("NCR:rt_ll_failed() \n"); 
#endif
	struct hdr_cmn *ch = HDR_CMN(p);
	struct hdr_ip *ih = HDR_IP(p);

	 drop(p, DROP_RTR_MAC_CALLBACK);
return;
}


// *****************
//  PACKET ROUTINES
// *****************


void
NCR::recv(Packet *p, Handler*) 
{
#ifdef NC_DEBUG
	printf("NCR:recv() \n"); 
#endif
	assert(initialized());

	assert(p);
	
	// If we are using flooding ...
	
	if (flooding) {

		recv_flooding(p);

	// If we want to use network cording ...
	} else {
		recv_NC(p);
	}
		
}


// *******************
// FOR NETWORG CODING
// The following functions are implemented to support all network coding functionalities
// *******************

/// **** Receiving function *******
/// This is the main function. It receives a packet from the demux and it has to manage the packet. It the source address corresponds to this node, it 
/// means that the packet is generated by this node so it has to be transmitted down. In particular if force first is true, each new generated packet is /// transmitted without coding after the inserction within the rigth buffer, while if force first is false, the packet is inserted within the buffer and /// if it is innovative a new random combination is sent.
/// On the contray, if the packet comes form the link layer and the destination address is IP_BROADCAST, the node has to receive the packet, intert it /// /// within the buffer and send out a new combination if it is the case.
/// *************

void NCR::recv_NC(Packet* p)
{

	assert(p);
	struct hdr_cmn *ch = HDR_CMN(p);
	struct hdr_ip *ih = HDR_IP(p);
	struct hdr_NC *nh = hdr_NC::access(p);

	if(pseudo == 1)
		neighbor = get_neighbor();
		

	//**************************
	// PACKET RECEPTION
	//**************************
	
	// CASE 1: 
	//---------
	//Packet comes from the application layer (it is a generated packet) and it has to be send down
	if (ih->saddr() == index_ && ch->ptype() != PT_NC_hello)
	{
	
#ifdef NC_DEBUG_final
		printf("NCR::recv (%f) Node %i recv my packet and send it \n", Scheduler::instance().clock(), index_);
#endif
		datasize_ = ch->size();

		
		sendMyPkt(p); // This function prepare the packet to be sent
		return;
	} 
	// If I receive an hello packet, I've update the stopping vector for that generation
	if(ch->ptype() == PT_NC_hello && ih->saddr() != index_ && stopping_strategy == 1){
		for(int i=0; i< num_buffers; i++){
			if(buffer[i]->generation() == nh->generation())
				buffer[i]->update_stop_vector(ih->saddr(), nh->rank());		
		}
		free(p);
		return;
	} else	if (ih->daddr() == IP_BROADCAST && ih->saddr() != index_) {
		
		// CASE 2: 
		// ---------
		//Packet comes from the link layer (it is a received packet) and it has to be processed

		struct hdr_ip *ih = HDR_IP(p);
			
		datasize_ = ch->size() - nh->lenCoeff() - NC_HDR_LEN;
		++statistic->app_recv;

#ifdef NC_DEBUG_final
		printf("NCR::recv (%f) Node %i, recv a packet from Node %i \n", Scheduler::instance().clock(), index_, ih->saddr());
		printf("NCR::recv: (%f) Nodo %i elaborate a NC packet from %i \n", Scheduler::instance().clock(), index_, ih->saddr());
		printf("\t generation %i \n", nh->generation());
		printf("\t lenght %i \n", nh->length());
		printf("\t size %i \n", ch->size());
		nh->printCoeff(nh->lenCoeff());
		printf("\t ID %i \n", ch->uid());
		printf("\t Matrix of the node \n");
#endif

		// It is inserted into the rigth buffer
		int buf_id = insert(p);
		bool new_n = buffer[buf_id]->set_new_neigh(ih->saddr());
 		
		update_vicini(ih->saddr());

		NCBuffer* buf = buffer[buf_id];
		assert(buf);

		// updating tau with the info coming with p
		double txtime = hdr_cmn::access(p)->txtime();
		double	Tdifs=0.00005;
		double T_s=txtime+Tdifs;
		int neighbors=get_num_vicini();
		tau_avg=T_s*neighbors;
		assert(tau_avg>0);
		// Update the stopping vector using the piggy backed stopping messages
		if(nh->full() == 1){
			buf->update_stop_vector(ih->saddr(), nh->rank());
		}

#ifdef NC_DEBUG
		printf("CONTENUTO DELLA LISTA GLOBALE \n");
		for (int i = 0; i<max_packets; i++) 
		{
		Packet* all = global_allocation.get_packet(i);
			if (all) {
				struct hdr_cmn *ch_all = HDR_CMN(all);
				struct hdr_ip *ih_all = HDR_IP(all);
				printf("\t i = %i, Packet %i, saddr %i, daddr %i \n", i, ch_all->uid(), ih_all->saddr(), ih_all->daddr());
			}
		}
#endif
		
		//**************************
		// STATISTICS
		//**************************
		// All statistics are collected

		if (statistics) {
			statistic->avg_cols += buf->_used_cols;
		
			if (buf->innovative()) 
			{
				double avg_delay = 0;
				int num_avg = 0;
				int new_decoded = 0;
				statistic->time_inn = CURRENT_TIME;
				++statistic->app_inn;
				for (int i = 0; i < NCBuffer::block_size; i++) 
				{
					Packet *gp = global_allocation.get_alloc(buf->generation(), i);
				
					if (gp) 
					{
						struct hdr_cmn *ch_gp = HDR_CMN(gp);
						struct hdr_ip *ih_gp = HDR_IP(gp);
						struct hdr_NC *nh_gp = hdr_NC::access(gp);
#ifdef NC_DEBUG
						printf("Extracted packet of Node %i - Node %i - index %i \n", ih_gp->saddr(), index_, i);
#endif
						// Ignore own packets
						if (ih_gp->saddr() != index_) 
						{
							//double delay = CURRENT_TIME - nh->timestamp();
							double delay = nh->timestamp() - nh_gp->timestamp();
#ifdef NC_DEBUG			
							printf("index_ %i Current Time: %f -- pkt: %i -- time_recv: %f -- time_insert: %f -- Delay %f \n", index_, CURRENT_TIME, i, nh->timestamp(), nh_gp->timestamp(), delay);
#endif						
							if (nh->coeff_i(i) != 0) 
							{
								// use average delay over all packets from other nodes
								// contained in the datagram
								++num_avg;
								//avg_delay += delay;
							//}
								// update stats for newly decodable packets
								if (buf->decoded(i) == 1) 
								{
									++new_decoded;
									avg_delay += delay;
									++statistic->app_decd;
									//statistic->app_dcdelay += delay;
									statistic->sum_dcdelay += delay;
								}								
							}
						}
					}
				}

				
				
				// innovative
				//++statistic->app_recv;
				assert(num_avg > 0 && avg_delay > 0);
				statistic->app_delay += avg_delay / num_avg;
				if (new_decoded > 0)
					statistic->app_dcdelay += avg_delay / new_decoded;
// 				file_delay = fopen(filename_delay,"a+");
// 				fprintf(file_delay,"%i %f %f %i \n", index_, statistic->time_inn, avg_delay/new_decoded, new_decoded);
// 				fclose(file_delay);
			} 
		}

		
		//**************************
		// SEND OUT A STOPPING MESSAGE
		//**************************
		// If the stopping strategy based on full matrix is active, when a node has a full rank it sends out its stopping message
		
		if (stopping_strategy == 1){
			// CASE 1: Send a Stop message when the matrix has a full rank
			if (full_matrix == 1 && buf->rank()>=N){
				if(buf->get_notify() < num_hello){
			#ifdef NC_DEBUG_fredi
			printf("NCR::recv (%f) Node %i  sends a stop message and has %i vicini \n", Scheduler::instance().clock(), index_,num_vicini);
			#endif
					send_hello(buf->generation(), buf->rank());
					buf->inc_notify();
				}
			}
		}

		// NB in case of Stopping strategy based on decoded packet, a node sends the Stop messages piggy backed or when the timer expires not at any receptions
		
		//**************************
		// SEND OUT A NEW COMBINATION
		//**************************
		// A new combination is sent if it is possible (i.e, the packet is innovative and the buffer priority is higher than zero).

		// If this is the first innovative reception the node starts to transmits
	
		if (buf->innovative() && stopping_strategy) {
			if (recv_inn == 0){
				Packet::free(p);
				double delay;
				int neighbors =get_num_vicini();
				if(neighbors < 3) neighbors = 3;
				delay = 2*NC_rng->genrand_real()*0.000752*(1.28*neighbors-3.55);
			recv_inn = recv_inn + 1;
				timer_.start(delay, 1);		
 			} 
		} else if (buf->innovative() && !stopping_strategy){
			Packet::free(p);
			if (timer_.isActive()) {
				recv_inn = recv_inn + 1;
				return;
			} else {
				double delay;
				delay = NC_rng->genrand_real()*timer_.get_interval();
				recv_inn = recv_inn + 1;
				timer_.start(delay, 0);
			}
		} else {
			Packet::free(p);
#ifdef NC_DEBUG_final		
			printf("Node %i: The received packet is not innovative \n", index_);
#endif
			}
	}
}

// *************************

// **** sendMyPkt ***
// This function sends a packet coming from the appliaction layer.
// It sets up the packet header layer and it decides if sending the packet with or without network coding on the base of the force_first parameter.
// STEPS:
// 1. Determine the packet generation
// 2. Create the packet setting also the coefficient
// 3. Insert the packet into the buffer
// 4. Collect Statistic
// *****************


void NCR::sendMyPkt(Packet *p) 
{
#ifdef NC_DEBUG
	printf("Node = %i, NCR:sendMyPkt() \n", index_); 
#endif

 
 	assert(initialized());

	// 1. Determine the generation of the packet (to be update)	
	
	int gen = getGeneration(generation_management);	
 	assert(gen >= 0);

	// 2. Set up the packet header

	struct hdr_cmn *ch = HDR_CMN(p);
	struct hdr_ip *ih = HDR_IP(p);
	struct hdr_NC *nh = hdr_NC::access(p);

	//Common header
	ch->size() = datasize_ + NC_HDR_LEN + NCBuffer::block_size; 
	ch->direction() = hdr_cmn::DOWN;
	ch->error() = 0;
	ch->ptype() = PT_NC;
	ch->addr_type() = NS_AF_NONE;
	ch->uid() = ++PktID;
	
	
	//IP header
	
	ih->saddr()	= index_;
	ih->daddr()	= IP_BROADCAST;
	ih->sport()	= ROUTER_PORT;
	ih->dport()	= ROUTER_PORT;

	nh->timestamp() = CURRENT_TIME;
#ifdef NC_DEBUG
	printf("\t ATTENZIONE: insert a packet of node %i in the global allocation \n", ih->saddr());
#endif
	int col = global_allocation.alloc_insert(p, index_, gen);
	

#ifdef NC_DEBUG
	printf("CONTENUTO DELLA LISTA GLOBALE \n");
	for (int i = 0; i< max_packets; i++) {
		Packet* all = global_allocation.get_packet(i);
		if (all) {
			struct hdr_cmn *ch_all = HDR_CMN(all);
			struct hdr_ip *ih_all = HDR_IP(all);
			printf("\t i = %i, Packet %i, saddr %i, daddr %i \n", i, ch_all->uid(), ih_all->saddr(), ih_all->daddr());
		}
	}
#endif
	assert(col < s);

	// Fill generation and column information in new packet

#ifdef USE_NTL	
	vec_GF2E content;
	content.SetLength(NCBuffer::block_size);
#else
	unsigned char *content = (unsigned char *) malloc(NCBuffer::block_size);
	bzero(content, NCBuffer::block_size);
#endif
	
	content[col] = 1;

#ifdef NC_DEBUG
	printf("Node = %i content: ", index_);
	for (int i = 0; i < NCBuffer::block_size; ++i) {	
		printf(" %i ", content[i]);
	}
	printf("\n");
#endif

	// NC header

	// Alloc the coefficients of the random combination

	nh->allocCoefficient(NCBuffer::block_size);
	nh->generation() = gen; 
	nh->length() = NCBuffer::block_size; 
	memcpy(nh->coefficient(), content, NCBuffer::block_size);
	
#ifdef NC_DEBUG_final
 	printf("\t Node = %i determine coefficient of the my packet: ", index_);
 	nh->printCoeff(NCBuffer::block_size);
 	printf("\n");
#endif
	free(content);
	nh->lenCoeff() = NCBuffer::block_size;
	
	if (pseudo == 1) {
		nh->pseudob() = 1;
		ch->next_hop() = neighbor;
	} else {
		ch->next_hop() = -1; 
		nh->pseudob() = 0;
	}
	
	// Piggy backed information for the stopping strategy

	nh->full() = 0;
	nh->rank() = 0;
#ifdef NC_DEBUG
 	printf("\t Node = %i Set up the headers \n", index_);
#endif

	// 3. Insert the new packet into the buffer
	int buf_id = insert(p);
	NCBuffer* buf = buffer[buf_id];
	assert(buf);
		
#ifdef NC_DEBUG
	printf("NCR::recv: (%f) Nodo %i elaborate a NC packet from %i \n", Scheduler::instance().clock(), index_, ih->saddr());
	printf("\t generation %i \n", nh->generation());
	printf("\t lenght %i \n", nh->length());
	printf("\t size %i \n", ch->size());
	printf("\t ID %i \n", ch->uid());
	printf("\t Chosen buffer %i \n", buf_id); 
#endif


	
	// 4. Send the packet

	// If force_first we send the new packet uncoded at least once
	// else we transmit a new random combination
	// NB With stopping strategy full decoded always force_first = 1

	if (force_first) {
#ifdef NC_DEBUG
		printf("Node = %i, NCR:sendMyPkt uncoded because force first using %i \n", index_, ch->next_hop()); 
#endif
		buf->inc_sendcount(); 
		++statistic->route_send;
		Scheduler::instance().schedule(target_, p, 0.0);
#ifdef NC_DEBUG_final	
		printf("\t Node = %i sent \n", index_);
#endif	
	} else if (buf->innovative()) {
#ifdef NC_DEBUG_final
		printf("Node = %i, NCR:sendMyPkt !force_first and innovative \n", index_); 
#endif
		// The cerated packet is inserted into the buffer but it will never be sent 
		Packet::free(p); 

		// While a new random combination is sent
		
		if(!stopping_strategy){
			if (timer_.isActive()) {
				recv_inn = recv_inn +1;
				return;
			} else {
				double delay;
				delay = NC_rng->genrand_real()*timer_.get_interval();
				recv_inn = recv_inn + 1;
				timer_.start(delay, 0);
			}
  		} else if (stopping_strategy && recv_inn ==0) {
			double delay;
			delay = NC_rng->genrand_real()*timer_.get_interval();
			recv_inn = recv_inn + 1;
			timer_.start(delay, 1);
		}
	}

// 5. Statistics
	++statistic->app_send;
}

// ************************************


// *** sendCombination ***
// This function send a new combination of packet 
// after the reception of an innovative packet or a new generated packet
// only if the buffer priority is greater than 0 or the stopping condition is not verifyied
// STEPS:
// 1. Find a buffer with the highest priority (if it exists)
// 2. Generate a new random combination to be sent
// 3. Send the packet
// ***********************

void NCR::sendCombination(double delay) 
{
#ifdef NC_DEBUG
	printf("NCR: sendCombination() \n"); 
#endif
	
	// Determine the buffer which has the higher priority to transmit a random combination 
	// of packet stored into it
	
	int tx_buffer = txBuffer();
	
#ifdef NC_DEBUG_final
	printf("NCR: sendCombination(), I have the buffer ID = %i \n", tx_buffer); 
#endif
	// Generate a new random combination
	if (tx_buffer < 0) {
		if(stopping_strategy && full_matrix == 1){
			timer_.stop();
	#ifdef NC_DEBUG_fredi
	printf("NCR::send (%f) Node %i, stopped its txes \n", Scheduler::instance().clock(), index_);
	#endif
	}	
		// In case of Stopping Strategy full decoded a node suspends its tranmissions but it can to continue if neighborhood situation changes
		
		// NB: Funziona solo per una generazione!
		if (full_decoded == 1 && buffer[0]->is_full_decoded() && count_hello < num_hello){
			//printf("Node %i send_hello buf_rank = %li \n", index_, buffer[0]->rank());
			send_hello(buffer[0]->generation(), buffer[0]->rank());
			buffer[0]->inc_notify();	
			++count_hello;	
		}
		return;
	} else {

		count_hello = 0; // Utility

		// A new combination from buffer buf is generated

		NCBuffer *buf = buffer[tx_buffer];

#ifdef NC_DEBUG
 		printf("BUFFER PRIORITY = %f \n", buf->priority(stopping_strategy)); 
#endif
		assert(buf && buf->priority(stopping_strategy) > 0);

		// The packet is prepared
		Packet *p = buf->generate();
		assert(buf->getGenerated());

#ifdef NC_DEBUG
 		printf("NCR: sendCombination(), I have generated the new combination \n"); 
#endif
	
		// Reset the combination
		buf->reset_generated();
	
		// Increase the number of transimtted packets (for probabilistic approach)

		buf->inc_sendcount();
			
		// Set up the header fields
			
		struct hdr_cmn *ch = HDR_CMN(p);
		struct hdr_ip *ih = HDR_IP(p);
		struct hdr_NC *nh = hdr_NC::access(p);	
		
		// Common header
		ch->uid() = ++PktID;
		ch->size() = ch->size() + datasize_; // Header IP + Header NC + Coefficient + datasize
		ch->direction() = hdr_cmn::DOWN;
		ch->error() = 0;
		ch->ptype() = PT_NC;
		ch->addr_type() = NS_AF_NONE;

		// IP header
		ih->saddr()	= index_;
		ih->daddr()	= IP_BROADCAST;
		ih->sport()	= ROUTER_PORT;
		ih->dport()	= ROUTER_PORT;
			
		// NC header
		nh->timestamp() = CURRENT_TIME;
	
	
		if (pseudo == 1) {
			nh->pseudob() = 1;
			ch->next_hop() = neighbor;	
		} else {
			ch->next_hop() = -1; 
			nh->pseudob() = 0;
		}

		// If neccesary include piggy backed information for the stopping strategy

		if (full_decoded == 1 && buf->is_full_decoded()){
			//printf("Node %i send_hello buf_rank = %li \n", index_, buf->rank());
			nh->full() = 1;
			nh->rank() = buf->rank();
		} else {
			nh->full() = 0;
			nh->rank() = 0;
		}
	
#ifdef NC_DEBUG
		printf("\t Packet=%i - timestamp = %f \n", ch->uid(), nh->timestamp());
#endif
		sendPkt_ = sendPkt_ + 1; 
		
#ifdef NC_DEBUG_final
		printf("NCR: sendCombination() Node %i has prepared the NEW CODED packet \n", index_); 
		printf("\t generation = %i - ", nh->generation());
		printf("\t length = %i -", nh->length());
		printf("\t size = %i -", ch->size());
		printf("\t Coefficient: ");
		nh->printCoeff(NCBuffer::block_size);
		printf("\n");
#endif

		// Send the packet
		assert(target_);
		assert(p);
		++statistic->route_send;
		Scheduler::instance().schedule(target_, p, delay);
		
#ifdef NC_DEBUG_FINAL
		printf("NCR::sendCombination: (%f) Nodo %i sends the new NC packet to %i using %i\n" , Scheduler::instance().clock(), index_, ih->daddr(), ch->next_hop()); 
#endif	
			
	}	
	return;
}


// *************
// AUXILIARY FUCTIONS
// *************

// *** insert ***
// This function inserts a new or received packet within one node's 
// buffer in the right position and returns the buffer id
// STEPS:
// 1. Find the right buffer (on the basis of the packet generation) and the right position.
// 2. Update the buffer's parameters such as the sendallowed and sendcount counter and the buffer priority
// **************

int NCR::insert(Packet *dg) 
{
#ifdef NC_DEBUG_final
	printf("NCR:insert() \n"); 
#endif
	// Pointers to the packet headers
	struct hdr_NC *nh = hdr_NC::access(dg);
	struct hdr_cmn *ch = HDR_CMN(dg);
	struct hdr_ip *ih = HDR_IP(dg);
	
	// Find the right buffer (buffer are divided by generations)
	// We compare the packet generation with the buffer generation. If there is a macth 
	//is good otherwise we create a new buffer

	int buf_id = 0;

	
	while (buf_id < num_buffers && !buffer[buf_id]->fits(dg))
	        ++buf_id;
	
	if (buf_id == num_buffers) 
	{
		++num_buffers;
		assert(num_buffers <= MAX_BUFFERS);
		// There is a maximum limit to the number of buffers 
		if (num_buffers > MAX_BUFFERS) 
		{
			fprintf(stderr, "Max number of buffer exceeded!\n");
			exit(-1);
		}

#ifdef USE_NTL	
		buffer[buf_id] = new NCBuffer(nh->generation(), nh->lenCoeff(), generation_size);
		buffer[buf_id]->set_stop_vector(N);
#else
		buffer[buf_id] = new NCBuffer(nh->generation(), nh->lenCoeff(), generation_size);
		buffer[buf_id]->set_stop_vector(N);
#endif
	}
	//printf("Insert len coeff % i, block s %i \n", nh->lenCoeff(), NCBuffer::block_size);
	NCBuffer* buf = buffer[buf_id];
	assert(buf);
#ifdef PRINT
	double prev = buf->priority(stopping_strategy);
#endif
	
	// Increment the recvcount of this buffer
	buf->inc_recvcount();
	// Insert the packet into the matrix
	buf->inject(dg);
	// Determine if transmit or not

	if (buf->innovative()) 
	{
		double inc = 0;
		double sc = calc_send_count(dg);
		if (probabilistic_nc && !wanted_strategy) 
		{
			double frac = modf(sc, &inc);
#ifdef NC_DEBUG_final
			printf("\t Node %i probabilistic: inc, frac %f inc %f \n", index_, frac, inc);	
#endif
			if (NC_rng->genrand_real() < frac) {
				inc += 1.0;
			}

		} else if (wanted_strategy){
 			if (nh->get_wanted());
				inc += 1.0;	
		} else if (!probabilistic_nc){
			inc = sc;
		}
		// send own packets at least once
		if (ih->saddr() == index_ && inc < 1){
			inc = 1.0;
			}
		buf->inc_sendallowed(inc);

#ifdef NC_DEBUG_final
	printf("NCR:insert inc=%f, buf_id=%i and sendallowed=%f \n", inc, buf_id, buf->sendallowed()); 
#endif
	}
#ifdef PRINT
	if (prev > 0 && buf->priority(stopping_strategy) == 0)
		write_log('d', dg);
#endif

	return buf_id;
}

// ******calc_send_count*******
// This function compute the value of send count which gives us the probability to increment or not the sendallowed counter 
// There are many different strategies:
// adaptive_send_count == 0 --> Normal Mode: fixed sendcount equal for each node.
// adaptive_send_count == 1 --> Neighbor based: the sendcount is equal to k/neigh where neigh is the number of node's negihbors and k a parameter chosen // by the user
// adaptive_send_count == 2 --> 2 hop Neighbor based: the sendcount is equal to k/min(2neigh) where 2neigh is the number of two hop negihbors and k a 
// parameter chosen by the user
// adaptive_send_count == 2 --> Last hop Neighbor based: the sendcount is equal to k/Lneigh where Lneigh is the number of negihbors of the current packet // source and k a parameter chosen by the user
// adaptive_send_count == 4 --> Distance based: the sendcount depends on the distance between the node and the current packet source
// adaptive_send_count == 5 --> Distance-neighbor based: the sendcount depends on the distance between the node and the current packet source and the 
// number of neighbors of the last hop (normalized version)
// adaptive_send_count == 6 --> Distance-neighbor based: the sendcount depends on the distance between the node and the current packet source and the 
// number of neighbors of the last hop (not normalized version)
// *************

double NCR::calc_send_count(Packet* p) 
{
#ifdef NC_DEBUG
	printf("NCR:calc_send_count \n"); 
#endif
	double s = send_count;
	
	if (adaptive_send_count == 0)
		s = send_count;
	
	if (adaptive_send_count == 1) {
		
		// Evaluate the number of neighbors
		
		int n_neigh = num_vicini;
		if (n_neigh > 1)
			s = s/n_neigh;
	} 
	
	if (adaptive_send_count == 2){
		
		int neighb = N;
		for(int i=0; i<N; i++){
			MobileNode* test_ = (MobileNode*)nodo->get_node_by_address(i);
			if(!test_==0 && i!=nodo->address() && nodo->distance(test_)<=R)
			{
				int neighb_tmp = num_neighbor(test_);
				if (neighb_tmp < neighb)
					neighb = neighb_tmp;
			}
	
		}
		s = s/neighb;
	} 
	if (adaptive_send_count == 3 || adaptive_send_count == 5 || adaptive_send_count == 6) {
		struct hdr_cmn *ch = HDR_CMN(p);
		struct hdr_ip *ih = HDR_IP(p);

		MobileNode* src = (MobileNode*)nodo->get_node_by_address(ih->saddr());
		int n_neigh = num_neighbor(src);
		if (n_neigh > 1 && adaptive_send_count != 6)
			s = s/n_neigh;
		else
			s = s*3/(2*n_neigh);
	} 
	
	if (adaptive_send_count >= 4) {
		struct hdr_cmn *ch = HDR_CMN(p);
		struct hdr_ip *ih = HDR_IP(p);
	
		MobileNode* src = (MobileNode*)nodo->get_node_by_address(ih->saddr());
		double d = nodo->distance(src);
		if (adaptive_send_count == 4){
			s = d/R;
		} else if (adaptive_send_count == 5){
			s = s*(1 + (d - R/2)/R);	
		} else if (adaptive_send_count == 6){
			s = s*(d/R);
		}
	}
	
	return s;
}

// ****** innovative *******
// This function determine if the packet is innovative or not for a specific buffer of a specific generation
// *************

bool NCR::innovative(Packet *p) 
{
#ifdef NC_DEBUG
	printf("NCR:innovative() \n"); 
#endif
	struct hdr_NC *nh = hdr_NC::access(p);
	struct hdr_cmn *ch = HDR_CMN(p);
	struct hdr_ip *ih = HDR_IP(p);
	
	assert(p);

	int i = 0;
	while (i < num_buffers && !buffer[i]->fits(p))
		++i;

	if (i == num_buffers)
		return true;

	NCBuffer *buf = buffer[i];
	if (buf->rank() == NCBuffer::block_size)
		return false;

#ifdef USE_NTL	
	mat_GF2E matrix = buf->matrix;
	matrix[buf->rank()] = nh->coefficient();
	int rank = gauss(matrix);
#else
	assert(nh->lenCoeff() == buf->_num_cols);
	unsigned char **matrix = (unsigned char **) malloc(NCBuffer::block_size * sizeof(unsigned char *));
	for (i = 0; i < NCBuffer::block_size; ++i) {
		matrix[i] = (unsigned char *) malloc(buf->_num_cols);
		memcpy(matrix[i], buf->matrix[i], buf->_num_cols);
	}
	memcpy(matrix, nh->coefficient(), nh->lenCoeff());
	int rank = GF2E::gauss(matrix, NCBuffer::block_size, nh->lenCoeff());
#endif
	return rank > buf->rank();
}

// ******get_matrix*******
// This function return the buffer number id
// *************

NCBuffer* NCR::get_matrix(int id) 
{
#ifdef NC_DEBUG
	printf("NCR:get_matrix(%i) \n", id); 
#endif
	if (id >= 0 && id < num_buffers)
		return buffer[id];
	return 0;
}

// ******getGeneration*******
// This function assings a generation number to a new generated packet. There are different generation management that the user can select:
// GEN_FIRST = 1
// GEN_RANDOM = 2 
// GEN_SMALLEST = 3
// GEN_NEAREST = 4
// GEN_GLOBAL = 5
// *************

int NCR::getGeneration(int gen_mng) 
{
#ifdef NC_DEBUG
	printf("NCR:getGeneration(%i) \n",gen_mng); 
#endif

	int gen = -1;

	// (GEN_FIRST = 1, GEN_RANDOM = 2, GEN_SMALLEST = 3, GEN_NEAREST = 4, GEN_GLOBAL = 5)
	
	int tmpoffset = 0, tmpsize = NCBuffer::block_size;
	double tmpdist = DBL_MAX;

	
	if (gen_mng == 2 && num_buffers > 1)
		tmpoffset = NC_rng->genrand_int32() % num_buffers;

	if (gen_mng == 5) {
		gen = global_allocation.last_generation();
		if (gen >= 0 && global_allocation.num_columns(gen) >= NCBuffer::block_size)
			gen = -1;
	} else {
		for (int i = 0; i < num_buffers; ++i) {
			NCBuffer *buf = buffer[(i + tmpoffset) % num_buffers];
			assert(buf);
				switch (gen_mng) {
				    case 2:
				    case 1:
					    if (gen == -1) {
						    gen = buf->_gen;
					    }
					    break;
				    case 3:
					    if (buf->_used_cols < tmpsize) {
						    gen = buf->_gen;
						    tmpsize = buf->_used_cols;
					    }
					    break;
				    case 4:
					    gen = buf->_gen;
					    break;
				    default:
					    assert(0);
				}
		}
	}

	// check if new generation is necessary
	if (gen == -1) {
		gen = global_allocation.max_generation() + 1;
	}
#ifdef NC_DEBUG
	printf("NCR:getGeneration return gen=%i \n",gen); 
#endif	
	return gen;

}

// *****txBuffer******
// This function determines the buffer from which the new packet is to be generated on the basis of buffer priority and the packet combination strategy in // use.
// Normal Strategy: based only on the priority of the buffers
// Simple strategy: if all neighbors of the nodes have decoded all packets none buffer id is returned (-1)
// Stopping strategy: a practical implementation of Simple strategy
// ***********

int NCR::txBuffer() 
{
#ifdef NC_DEBUG
	printf("NCR:txBuffer() \n"); 
#endif

	
	tx_buffer = -1;
	for (int i = 0; i < MAX_BUFFERS; ++i)
		buf_checked[i] = 0;

	// iterate through the buffers
	double maxp = 0.0;
	int tb = -1;
	for (int i = 0; i < num_buffers; ++i) {
		assert(buffer[i]);
		if(simple_strategy){
			double p_tmp = buffer[i]->priority(0);
			if (buf_checked[i] == 0 && p_tmp > maxp && check_rank_neigh(buffer[i]->generation())) {
				maxp = p_tmp;
				tb = i;
			}
		} 
		if(!simple_strategy && !stopping_strategy) {
			if (buf_checked[i] == 0 && buffer[i]->priority(0) > maxp) {
				maxp = buffer[i]->priority(0);
				tb = i;
			}
		}
		if(stopping_strategy) {
			double p_tmp = buffer[i]->priority(1);
			if (buf_checked[i] == 0 && p_tmp > maxp) {
				maxp = p_tmp;
				tb = i;
			}
		}
	}

	if (tb >= 0) {
		tx_buffer = tb;
		buf_checked[tb] = 1;
		return tx_buffer;
	} else {
		return -1;
	}
}

void NCR::init() {

// Inizializations
	 max_packets = 2*NCBuffer::block_size*100;
	// init RNG
	if (seed == 0) {
		struct timeval t;
		gettimeofday(&t, 0);
		int pid = getpid();
		seed = (int) t.tv_usec ^ pid;
	}
	NC_rng = new RNG_NC();
	NC_rng->init_genrand(seed);
	
	for (int r=0; r < index_*10 + scenario; r++)
		NC_rng->genrand_real();
	
	// Init of the buffer
	NCBuffer::init(generation_size);
	// initialize global allocation table with max. number of app generated packets
	// HACK: use factor of 2*NCBuffer::block_sizeize for safety since number of packets is not deterministic
	// and theoretically each packet could be in its own generation
	//long max_packets = 2*NCBuffer::block_sizei*100;

	// Init of the global allocation
	if (index_==0){
		global_allocation.init(max_packets);
	}

	return;
}

void
NCR::trace (char *fmt,...)
{
	va_list ap;

	if (!tracetarget)
		return;
	
	va_start (ap, fmt);
	vsprintf (tracetarget->pt_->buffer (), fmt, ap);
	tracetarget->pt_->dump ();
	va_end (ap);
}

// *****send_Hello******
// This function is used to send a message to node's neighbor and advertise that the node has decoded all it wants
// ***********

void NCR::send_hello(int g, long int r){
	Packet* p = Packet::alloc();

	struct hdr_cmn *ch = HDR_CMN(p);
	struct hdr_ip *ih = HDR_IP(p);
	struct hdr_NC *nh = hdr_NC::access(p);

	// Common header
	ch->size() = datasize_ + NC_HDR_LEN; 
	ch->direction() = hdr_cmn::DOWN;
	ch->error() = 0;
	ch->ptype() = PT_NC_hello;
	ch->addr_type() = NS_AF_NONE;
	ch->uid() = ++PktID;
	
	
	// IP header
	ih->saddr()	= index_;
	ih->daddr()	= IP_BROADCAST;
	ih->sport()	= ROUTER_PORT;
	ih->dport()	= ROUTER_PORT;

	nh->generation() = g; 
	nh->length() = 0; 
	nh->lenCoeff() = 0;
	nh->timestamp() = CURRENT_TIME;
	nh->full() = 1;
	nh->rank() = r;

	assert(target_);
	assert(p);
	++statistic->hello_send;
	double delay_t = 0.0;
	if (full_matrix == 1)
		delay_t = NC_rng->genrand_real()*0.002;
	
	Scheduler::instance().schedule(target_, p, delay_t);
}



// **********************
// FOR FLOODING
// The following functions are implemented to support all network coding functionalities
// **********************

// **** Receiving function *******
// This is the function used to send packets without using network coding. It receives a packet from the demux and it has to manage the packet. It the 
// source address corresponds to this node, it 
// means that the packet is generated by this node so it has to be transmitted down. 
// On the contray, if the packet comes form the link layer and the destination address is IP_BROADCAST, the node has to receive the packet.
// *************


void NCR::recv_flooding(Packet* p) {
	
	struct hdr_cmn *ch = HDR_CMN(p);
	struct hdr_ip *ih = HDR_IP(p);
	struct hdr_NC *nh = hdr_NC::access(p);


	neighbor = get_neighbor();


	if (ih->saddr()==index_ && !seqno_known(ch->uid())){

#ifdef NC_DEBUG_final
		printf("NCR::recv (%f) Node %i, recv my packet and send it \n", Scheduler::instance().clock(), index_);
#endif
		
		// Common header
		ch->size() = datasize_ + NC_HDR_LEN + NCBuffer::block_size; 
		ch->direction() = hdr_cmn::DOWN;
		ch->error() = 0;
		ch->ptype() = PT_NC;
		ch->addr_type() = NS_AF_NONE;
		ch->next_hop() = IP_BROADCAST;
		//ch->uid() = ++PktID;
	
		// IP header
		ih->saddr()	= index_;
		ih->daddr()	= IP_BROADCAST;
		ih->sport()	= ROUTER_PORT;
		ih->dport()	= ROUTER_PORT;

		// NC header
		nh->generation() = 0; 
		nh->length() = NCBuffer::block_size; 
		nh->lenCoeff() = 0;
		nh->timestamp() = CURRENT_TIME;
		++statistic->app_send;
		seqno_insert(ch->uid());
		if (pseudo == 1) {
			nh->pseudob() = 1;
			ch->next_hop() = neighbor;
		} else {
			ch->next_hop() = -1; 
			nh->pseudob() = 0;
		}
		Scheduler::instance().schedule(target_, p, 0.0);
	}
		
	if (ih->saddr()!=index_){

		struct hdr_ip *ih = HDR_IP(p);
		

		bool del = false;
		bool app_received = false;
	
		assert(p);

		if (seqno_known(ch->uid())) {
			++statistic->app_recv;
			del = true;
		} else {
			if (ih->daddr() == index_ ) {
				app_received = true;
				++statistic->app_recv;
				del = true;
			
			} else if (ih->daddr()==IP_BROADCAST) {
				++statistic->app_recv;
				app_received = true;
			}
		
			seqno_insert(ch->uid());
		}
		
		if (statistics) {
			if (app_received) {
				statistic->time_inn = CURRENT_TIME;
				++statistic->app_decd;
				//statistic->app_hops += channel->cost(p->saddr_, id);
				statistic->app_delay += CURRENT_TIME - nh->timestamp();
				statistic->app_dcdelay += CURRENT_TIME - nh->timestamp();
			}
		}
		
		if (del) {
			Packet::free(p);
		} else {
			double sc = calc_send_count(p);
			if (NC_rng->genrand_real() < sc) {
#ifdef NC_DEBUG_final
			printf("\t Node %i probabilistic flood ok \n", index_);	
#endif
				ch->direction() = hdr_cmn::DOWN;
				ih->saddr() = index_;
				if (pseudo == 1) {
					nh->pseudob() = 1;
					ch->next_hop() = neighbor;
				} else {
					ch->next_hop() = -1; 
					nh->pseudob() = 0;
				}
				Scheduler::instance().schedule(target_, p, 0.0);
			} else {
				Packet::free(p);
			}
			// enqueue copies if necessary
			if (sc > 1.0) {
				int red = (int) sc;
				if (NC_rng->genrand_real() < sc - red)
					++red;
				for (; red > 1; --red) {
					Packet *c = p->copy();
					struct hdr_cmn *ch = HDR_CMN(c);
					ch->direction() = hdr_cmn::DOWN;
					ih->saddr() = index_;
					if (pseudo == 1) {
						nh->pseudob() = 1;
						ch->next_hop() = neighbor;
					} else {
						ch->next_hop() = -1; 
						nh->pseudob() = 0;
					}
					Scheduler::instance().schedule(target_, c, 0.0);
				}
			}
		}
	}
}

int NCR::seqno_known(int n) {
	assert(n >= 0);
	if (n < seqno_low)
		return INT_MAX;
	if (n >= seqno_low + SEQNO_CACHESIZE)
		return 0;
	// cache wraps around, starts at index seqno_idx with actual seqno seqno_low
	return seqno_cache[(seqno_idx + n - seqno_low) % SEQNO_CACHESIZE];
}

void NCR::seqno_insert(int n) {
	// seq no not relevant anymore or fits into array
	if (n < seqno_low + SEQNO_CACHESIZE) {
		if (n >= seqno_low)
			seqno_cache[(seqno_idx + n - seqno_low) % SEQNO_CACHESIZE] += 1;
		return;
	}

	// shift array bounds to make seq no fit (shift = n - last known)
	int last_seq = seqno_low + SEQNO_CACHESIZE - 1;	
	int shift = n - last_seq;
	seqno_low += shift;
	for (int i = 1; i < shift; ++i) {
		// set entries between previously last and new one to "false"
		seqno_cache[seqno_idx++] = 0;
		if (seqno_idx == SEQNO_CACHESIZE)
			seqno_idx = 0;
	}
	// seqno_idx++ once more so that it jumps from the last to the first entry
	seqno_cache[seqno_idx++] = 1;
	if (seqno_idx == SEQNO_CACHESIZE)
		seqno_idx = 0;

	return;
}

// **************************
// TOPOLOGY FUNCTIONS
// These functions are useful to gather some information about the node's neighborhood
// **************************

int NCR::get_neighbor() {	
	int i = (int)round(NC_rng->genrand_real()*num_vicini);
	return vicini[i];
}

int NCR::num_neighbor(MobileNode* nn) {
	int neighb = 0;
	for(int i=0; i < N; i++){
		MobileNode* test_ = (MobileNode*)nn->get_node_by_address(i);
		if(!test_== 0 && i!=nn->address() && nn->distance(test_)<=R)
		{
			neighb = neighb + 1;
		}
	}
	return neighb;
}


bool NCR::check_rank_neigh(long int g){
	for(int i = 0; i < num_vicini; i++){
		MobileNode* nc_neigh = (MobileNode*)nodo->get_node_by_address(vicini[i]);
		Tcl& tcl = Tcl::instance();
		tcl.evalf("$node_(%i) eval_rank %li", i, g);
		long int rank= atol(tcl.result());	
		if(rank < N){			
			return true;
		}
	}
	return false;
}

long int NCR::get_rank_of_g(long int g){
	long int rank_tmp = 0;
	for(int b = 0; b < num_buffers; b++){
		if(buffer[b]->generation() == g){
			rank_tmp = buffer[b]->rank();
			break;
		}	
	}
	return rank_tmp;
}

bool NCR::update_neigh(){
	num_vicini = num_neighbor(nodo);
	for (int i = 0; i < 100; i++)
		vicini[i] = 0;
	int count = 0; 

	for(int i = 0; i< N; i++){
		MobileNode* test_ = (MobileNode*)nodo->get_node_by_address(i);
		if(!test_== 0 && i!=nodo->address() && nodo->distance(test_)<=R)
		{
			vicini[count] = i;
			count++;
		}
	}
	return true;
}

void NCR::update_vicini(int n){

// 	bool ok = true;
// 	int count = 0;
// 	for (int i = 0; i < num_vicini; i++)	
// 		if (vicini[i]==n)
// 			ok = false;
// 	
// 	if(ok)
// 		++num_vicini;
//HACK bug

bool found=false;
        for(int i =0; i <= num_vicini; i++){
                if (vicini[i]==n)
                found=true;
                }
        if(!found){
                vicini[num_vicini]=n;
                num_vicini++;
        }
}

