/*
 * Copyright (c) 2009 Regents of the SIGNET lab, University of Padova and DOCOMO Communications Laboratories Europe GmbH.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Padova (SIGNET lab) nor the 
 *    names of its contributors may be used to endorse or promote products 
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED 
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/// This module is a networking layer module (routing) which is able to route packets and perform random netowrk coding.


#include <agent.h>
#include <packet.h>
#include <sys/types.h>
//#include <cmu/list.h>
#include <scheduler.h>

#include <cmu-trace.h>
#include <priqueue.h>
#include <classifier/classifier-port.h>
#include <mac.h>
#include <mobilenode.h>

#include "nc_buffer.h"
#include "trace.h"
#include "rng_NC.h"

#define NC_HDR_LEN 18
#define MAX_BUFFERS 1000
#define ROUTER_PORT 0xff
#define SEQNO_CACHESIZE 20000

//#define HDR_NC(p) (hdr_NC::access(p))

enum gen_mgmt_type {GEN_FIRST, GEN_RANDOM, GEN_SMALLEST, GEN_NEAREST, GEN_GLOBAL};


///
///  NCR header: This is the network coding packet header which contains the random coefficients of the packet combination
///

typedef struct hdr_NC {
	 
	int len_;			/// \brief Lenght of the data field
	long gen_; 			/// \brief Generation of the packet
	unsigned char* coefficient_;	/// \brief Coefficient of the random combination
	unsigned int lenCoeff_; 	/// \brief Length of the coefficients
	double time_;			/// \brief Timestamp of the generation of the packet
	int pseudo_broadcast_;		/// \brief Field toactivate the pseudo broadcast mechanisms
	int full_;			/// \brief I can decoded all in my buffer for this generation
	long int rank_;			/// \brief Current rank of this generation in my buffer
	int want_;

	// Methods to access to the header fields
	inline	long& generation() { return (gen_);}
	inline	int& length() { return (len_);}
	inline	double& timestamp() { return (time_);}
	inline	int& pseudob() { return (pseudo_broadcast_);}
	inline 	int& get_wanted() {return (want_);}
	inline	unsigned char* coefficient() { return (coefficient_);}
	inline 	unsigned char& coeff_i(int i) {return (coefficient_[i]);}
	inline	unsigned int& lenCoeff() { return (lenCoeff_);}
	inline	int& full() { return (full_);}
	inline	long int& rank() { return (rank_);}
	
	// Inizialization of the coefficient field
	inline  void allocCoefficient(unsigned int n) {
		coefficient_ = (unsigned char *) malloc(n);
	}
	inline void copyCoeff(unsigned char* c, int size) {
		memcpy(&coefficient_, c, size);
	}
	inline void printCoeff(int size) {
		for (int i = 0; i < size; ++i) {	
			printf("%2x ", coefficient_[i]);
		}
		printf("\n");
	}
	
	// Access to the NC header
	static int offset_;
	inline static int& offset() { return offset_; }
	inline static struct hdr_NC* access(const Packet* p) {
		return (struct hdr_NC*)p->access(offset_);
	}

} hdr_NC;


// Class for the collecction of the statistics

class NodeStats;

class NodeStats {
public:
	NodeStats() : mac_send(0), mac_recv(0), app_send(0), app_recv(0),
		      app_decd(0), app_delay(0), app_dcdelay(0), route_send(0), app_hops(0), node_energy(0), avg_cols(0), sum_dcdelay(0), time_inn(0), app_inn(0), hello_send(0) {}

	int mac_send;
	int mac_recv;
	int app_send;
	int app_recv;
	int app_decd;
	int app_inn;
	int route_send;
	int hello_send;
	double app_delay;
	double app_dcdelay;
	double sum_dcdelay;
	int app_hops;
	double node_energy;
	int avg_cols;
	double time_inn;

};

class NCR;

/**
*  Protocol TIMER handler. It is use to manage the transmission of the packets and the waiting time
*/

class NCR_timer : public TimerHandler {
public:
	NCR_timer(NCR* agent);
	virtual void start(double delay, int s);
	double get_interval(){ return interval_;}
	virtual void stop()
	{
		stop_time = true;
		force_cancel();
	}
	virtual int isActive() 
	{
		return active_;
	} 
protected:
	NCR*	agent_;	
	virtual void expire(Event* e);

	double interval_;
	int active_;
	int stopping_s;	
	bool stop_time;
};

/**
* The main class of Random Network Coding Package is the NCR class. NCR is an Agent object and it is seen in the
* ns2 context as a routing layer. To attach this Agent to a node, the user has to follow the same procdure used for
* the other routing layer (i.e., AODV, DSR, ...).
*
*/

class NCR: public Agent {
  
 public:
        NCR(nsaddr_t id);
        void		recv(Packet *p, Handler *);
	void 		ncr_rt_failed_callback(Packet *p, void *arg); 
	void            rt_ll_failed(Packet *p);
        void            handle_link_failure(nsaddr_t id);
	void 		init();
	/** 
	* This function send a new combination of packet after the reception of an innovative packet or a new generated packet only if the buffer priority * is greater than 0 or the stopping condition is not verifyied
	* STEPS:
	* 1. Find a buffer with the highest priority (if it exists)
	* 2. Generate a new random combination to be sent
	* 3. Send the packet
	*/
	void 		sendCombination(double delay);
	int 		get_recv_inn(){return recv_inn;}
	void 		set_recv_inn(int i) {recv_inn = i;}
	double 		get_sendcount(){return send_count;}
	double tau_avg;
	double 	get_tau(){return tau_avg;}
	
	/** Get the current value of the innovative rate*/
	double get_innovative_rate(){return innovative_rate;}
	int get_num_vicini(){return num_vicini;}
	RNG_NC *NC_rng;

 protected:
        int             command(int, const char *const *);
        int             initialized() { return 1 && target_; }
	void		trace(char *fmt, ...);

	/**
	* This is the function used to send packets without using network coding (i.e, probabilistic flooding). It receives a packet from the demux and it has to manage the packet. It * the source address corresponds to this node, it means that the packet is generated by this node so it has to be transmitted down. 
	* On the contray, if the packet comes form the link layer and the destination address is IP_BROADCAST, the node has to receive the packet.
	*/
	void 		recv_flooding(Packet* p);
	/**
	* This is the main function. It receives a packet from the demux and it has to manage the packet. It the source address corresponds to this node, * it means that the packet is generated by this node so it has to be transmitted down. In particular if force first is true, each new generated 
	* packet is transmitted without coding after the inserction within the rigth buffer, while if force first is false, the packet is inserted within * the buffer and if it is innovative a new random combination is sent.
	* On the contray, if the packet comes form the link layer and the destination address is IP_BROADCAST, the node has to receive the packet, intert 
	* it within the buffer and send out a new combination if it is the case.
	* It is divided into two parts:
	* CASE 1: Packet comes from the application layer (it is a generated packet) and it has to be send down
	* CASE 2: Packet comes from the link layer (it is a received packet) and it has to be processed
	* 	  - It is inserted into the rigth buffer
	*	  - All statistics are collected
	*	  - A new combination is sent if it is possible (i.e, the packet is innovative and the buffer priority is higher than zero).
	*/
	void 		recv_NC(Packet* p);
	/** 
	* This function sends a packet coming from the appliaction layer.
	* It sets up the packet header layer and it decides if sending the packet with or without network coding on the base of the force_first parameter.
	* STEPS:
	* 1. Determine the packet generation
	* 2. Create the packet setting also the coefficient
	* 3. Insert the packet into the buffer
	* 4. Collect Statistic
	*/
	void            sendMyPkt(Packet *p);
	/**
	* This function is used to send a message to node's neighbor and advertise that the node has decoded all it wants
	*/ 
        void		send_hello(int g, long int r);
	/**
	* This function inserts a new or received packet within one node's 
	* buffer in the right position and returns the buffer id
	* STEPS:
	* 1. Find the right buffer (on the basis of the packet generation) and the right position.
	* 2. Update the buffer's parameters such as the sendallowed and sendcount counter and the buffer priority
	*/
	int 		insert(Packet *dg);
	Packet* 	createPkt(unsigned int s, long g);	
	void            sendNewPkt(int len, int address);
	
	/**
	* This function determine if the packet is innovative or not for a specific buffer of a specific generation
	*/
	bool 		innovative(Packet *p);
	NCBuffer* 	get_matrix(int id);

	/**
	* This function compute the value of send count which gives us the probability to increment or not the sendallowed counter 
	* There are many different strategies:
	* adaptive_send_count == 0 --> Normal Mode: fixed sendcount equal for each node.
	* adaptive_send_count == 1 --> Neighbor based: the sendcount is equal to k/neigh where neigh is the number of node's negihbors and k a parameter chosen by the user
	* adaptive_send_count == 2 --> 2 hop Neighbor based: the sendcount is equal to k/min(2neigh) where 2neigh is the number of two hop negihbors and k * a parameter chosen by the user
	* adaptive_send_count == 2 --> Last hop Neighbor based: the sendcount is equal to k/Lneigh where Lneigh is the number of negihbors of the current * packet source and k a parameter chosen by the user
	* adaptive_send_count == 4 --> Distance based: the sendcount depends on the distance between the node and the current packet source
	* adaptive_send_count == 5 --> Distance-neighbor based: the sendcount depends on the distance between the node and the current packet source and 
	* the number of neighbors of the last hop (normalized version)
	* adaptive_send_count == 6 --> Distance-neighbor based: the sendcount depends on the distance between the node and the current packet source and 
	* the number of neighbors of the last hop (not normalized version)
	*/
	double 		calc_send_count(Packet* p);
	/**
	* This function assings a generation number to a new generated packet. There are different generation management that the user can select:
	* GEN_FIRST = 1
	* GEN_RANDOM = 2 
	* GEN_SMALLEST = 3
	* GEN_NEAREST = 4
	* GEN_GLOBAL = 5
	*/ 
	int 		getGeneration(int gen_mng);
	/**
	* This function determines the buffer from which the new packet is to be generated on the basis of buffer priority and the packet combination 
	* strategy in use.
	* Normal Strategy: based only on the priority of the buffers
	* Simple strategy: if all neighbors of the nodes have decoded all packets none buffer id is returned (-1)
	* Stopping strategy: a practical implementation of Simple strategy
	*/
	int 		txBuffer(); 
	int 		get_neighbor();
	int 		num_neighbor(MobileNode* nn);
	int 		seqno_known(int n);
	void 		seqno_insert(int n);

	bool check_rank_neigh(long int g); 
	long int get_rank_of_g(long int g); 
        
	bool update_neigh(); 
 	void update_vicini(int n);
	void update_buffer_stop_vector(); 
	// NCR varaibles
        nsaddr_t    index_;	// IP Address of this node
        MobileNode *nodo;	// This node
	int N; 			// number of nodes
	int R;
	int sendPkt_; 		// Number of sent packet
	
	int datasize_;
	int recv_inn;
	
	// Buffer variables
	NCBuffer* buffer[MAX_BUFFERS];
	int buf_checked[MAX_BUFFERS];
	int num_buffers;
	int tx_buffer;

	int seqno_cache[SEQNO_CACHESIZE];
	int seqno_low;
	int seqno_idx;	
	
	// Statistic
	NodeStats *statistic;
	bool statistics;

	// Default parameters (!! do not modifiy !!)
	// NB: These parameters are setted from tcl 

 	int c_c; 
 	int p_n;
 	int f_f;
 	int st;
	/// \brief To activate channel coding
	bool channel_coding;  	
	/// \brief To activate probabilistic network coding	
	bool probabilistic_nc; 	
	/// \brief To activate force first	
	bool force_first; 
		
	int ncbs_hop_count; 
	/// \brief To activate simple probabilistic flooding	
	int flooding; 		
	/// \brief To activate the pseudo broadcast mechanism	
	int pseudo;  			
	int topology;
	int topo_dimension;
	int neighbor; 
	int scenario;
	/// \brief To specify the generation management to use
	int generation_management; 
	/// \brief To set up the k parameter	
	double send_count; 	
	/// \brief To select the forwarding factor strategies	
	double adaptive_send_count; 	
	/// \brief To activate the simple strategy
	int simple_strategy; 	
	int wanted_strategy; 
	
	/// \brief To activate the stopping strategy	
	int stopping_strategy; 	
	/// \brief To specify the number of stop messages	
	int num_hello; 			
	/// \brief To activate stopping strategy based on the knowledge about the number of required packets
	int full_matrix;
	/// \brief To activate stopping strategy based on the possibility to decoded all stored packets or not
	int full_decoded;
	int count_hello;

	/// \brief Innovative rate is the rate at which the node receives innovative packets	
	double innovative_rate;
	double last_innovative;

 	FILE* fileout;
 	char filename[70];
	FILE* file_inn;
 	char filename_inn[70];

	FILE* file_topo;
 	char filename_topo[70];

// 	FILE* file_delay;
//  	char filename_delay[70];

	/// \brief To specify the generation size
	int generation_size; 		
	
	/*
         * A mechanism for logging the contents of the routing
         * table.
        */
        Trace *tracetarget;   
	
	/*
         * A pointer to the network interface queue that sits
         * between the "classifier" and the "link layer".
         */
        PriQueue        *ifqueue;

	/* For passing packets up to agents */

	PortClassifier *dmux_;
	NsObject *port_dmux_;     // my port dmux
	int use_mac_;

	NCR_timer timer_;

	int num_vicini;
	int vicini[100];
	
};


