/*
 * Copyright (c) 2009 Regents of the SIGNET lab, University of Padova and DOCOMO Communications Laboratories Europe GmbH.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Padova (SIGNET lab) nor the 
 *    names of its contributors may be used to endorse or promote products 
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED 
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


/*
GF2E is the class to manage all the mathematical operations in a finite Galois field.
*/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include "gf2e.h"

unsigned char GF2E::GF2E_E[256];
unsigned char GF2E::GF2E_L[256];

// 0x1b = x^8 + x^4 + x^3 + x + 1 (AES)
unsigned char GF2E::ffmul(unsigned char a, unsigned char b) {
	unsigned char aa = a, bb = b, r = 0, t;
	while (aa != 0) {
		if ((aa & 1) != 0)
			r = r ^ bb;
		t = bb & 0x80;
		bb = bb << 1;
		if (t != 0)
			bb = bb ^ 0x1b;
		aa = aa >> 1;
	}
	return r;
}

void GF2E::init() {
#ifdef NC_DEBUG
	printf("GF2E:init() \n"); 
#endif
	int index = 0;
	unsigned char x = 0x01;
	GF2E_E[index++] = 0x01;
	for (int i = 0; i < 256; i++) {
		unsigned char y = ffmul(x, 0x03);
		GF2E_E[index++] = y;
		x = y;
	}

	for (int i = 0; i < 256; i++) {
		GF2E_L[GF2E_E[i]] = (unsigned char)i;
	}
}

unsigned char GF2E::mul(unsigned char a, unsigned char b) {
	int t = 0;
	if (a == 0 || b == 0)
		return 0;
	t = GF2E_L[a] + GF2E_L[b];
	if (t > 255)
		t -= 255;
	return GF2E_E[t];
}

unsigned char GF2E::inv(unsigned char a) {
#ifdef NC_DEBUG
	printf("GF2E:inv() \n"); 
#endif
	assert(a != 0);
	unsigned char t = 0xff - GF2E_L[a];
	return GF2E_E[t];
}

int GF2E::gauss(unsigned char **m, int rows, int cols) {
#ifdef NC_DEBUG
	printf("GF2E:gauss() \n"); 
#endif
	// assume matrix is "gaussian eliminated" except for last row
	int r, c;
	int rank = 0;

	for (c = 0, r = 0; c < cols && r < rows-1; ++c) {
		if (m[r][c] != 0) {
			++rank;
			++r;
		}
	}

	for (c = 0, r = 0; c < cols; ++c) {
		if (m[rows-1][c] != 0) {
			 //make first entry a 1
			if (m[rows-1][c] != 1) {
				unsigned char inverse = inv(m[rows-1][c]);
				//printf("Inverse = %2x \n", inverse);
				for (int i = c; i < cols; ++i){
					m[rows-1][i] = mul(m[rows-1][i], inverse);
				//	printf("%2x ", m[rows-1][i]);
				}
				//printf("\n");
				assert(m[rows-1][c] == 1);
			}
			
			if (r == rows - 1) {
				// last row already in place
				++rank;
				break;
			} else if (m[r][c] == 0) {
				// insert last row at proper place
				unsigned char *tmp = m[rows-1];
				int i;
				for (i = rows-1; i > r; --i)
					m[i] = m[i-1];
				m[i] = tmp;
				++rank;
				break;
			} else {
				assert(m[r][c] == 1);
				for (int i = c; i < cols; ++i)
					m[rows-1][i] ^= m[r][i];
				assert(m[rows-1][c] == 0);
				++r;
			}
		} else if (m[r][c] != 0) {
			++r;
		}
	}

	return rank;
}
