/*
 * Copyright (c) 2009 Regents of the SIGNET lab, University of Padova and DOCOMO Communications Laboratories Europe GmbH.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Padova (SIGNET lab) nor the 
 *    names of its contributors may be used to endorse or promote products 
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED 
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

// ********************
// **** BUFFER class ***
// This module contains all the structures and methods to store packets within a node and support network coding functionalities 
// ********************

/* -*-	Mode:C++; c-basic-offset:8; tab-width:8; indent-tabs-mode:t -*- */


#include <ip.h>
#include <assert.h>
#include <strings.h>
#include <limits.h>
#include <math.h>
#include <random.h>
#include <cmu-trace.h>
#include <energy-model.h>
#include <ll.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/time.h>

// XXX for redundancy_factor

#include "rng_NC.h"
#include "nc_buffer.h"
#include "NCR.h"

#define CURRENT_TIME    Scheduler::instance().clock()
//#define NC_DEBUG
//#define NC_DEBUG_final
//extern RNG_NC* rng;

int NCBuffer::opt_fullrankfw = 0;
int NCBuffer::block_size = 100;
int NCBuffer::target_size = 0;
int NCBuffer::field_size = 8;
int NCBuffer::abs_send_count = INT_MAX;
double NCBuffer::genvec_maxentries = 0;
static int rngB_initialized = 0;
int NCBuffer::recv_count = 0;

//
// Contructors and Initialization
//

void NCBuffer::init(long int b) {
	
	NCBuffer::block_size = b;
	// range check for target size of matrix
	if (NCBuffer::target_size <= 0 || NCBuffer::target_size >= NCBuffer::block_size)
		NCBuffer::target_size = NCBuffer::block_size;
		
#ifdef USE_NTL	
	GF2X seed;
	BuildIrred(seed, field_size);
	GF2E::init(seed);
	GF2X::HexOutput = 1;
#else
	GF2E::init();
	assert(NCBuffer::field_size == 8);
#endif

}

NCBuffer::NCBuffer(long g, int d, long int b) {

	_gen = g;
	_sendcount = 0;
	_sendallowed = 0;
	_recvcount = 0;
	_rank = 0;
	_innovative = false;
	_num_cols = d;
	_used_cols = 0;
	generated_ = 0;
	generated_rank_ = 0;
	already_notify = 0;
	

	NCBuffer::block_size = b;
	init(b);

	NC_rngB = new RNG_NC();
	struct timeval t;
	gettimeofday(&t, 0);
	int pid = getpid();
	int seed_B = (int) t.tv_usec ^ pid;
	NC_rngB->init_genrand(seed_B);

	_decoded = new int[block_size];
	for (int i = 0; i < block_size; ++i)
		_decoded[i] = 0;

	decoded_pkt = 0;
	full_decoded = false;
	// initialize matrix

#ifdef USE_NTL	
	matrix.SetDims(block_size, _num_cols);
#else
	matrix = (unsigned char **) malloc(block_size * sizeof(unsigned char *));
	for (int i = 0; i < block_size; ++i) {
		matrix[i] = (unsigned char *) malloc(_num_cols);
		bzero(matrix[i], _num_cols);
	}
#endif

}



NCBuffer::NCBuffer(const NCBuffer &b) {
	_gen = b._gen;
	_sendcount = b._sendcount;
	_sendallowed = b._sendallowed;
	_recvcount = b._recvcount;
	_rank = b._rank;
	_innovative = false;
	_num_cols = b._num_cols;
	_used_cols = b._used_cols;
	generated_ = 0;
	generated_rank_ = 0;
	_decoded = new int[block_size];
	for (int i = 0; i < block_size; ++i)
		_decoded[i] = b._decoded[i];
	decoded_pkt = 0;
	full_decoded = false;
	already_notify = 0;
	recv_count = 0;
	
// initialize matrix

#ifdef USE_NTL	
	matrix.SetDims(block_size, _num_cols);
	matrix = b.matrix;
#else
	matrix = (unsigned char **) malloc(block_size * sizeof(unsigned char *));
	for (int i = 0; i < block_size; ++i) {
		matrix[i] = (unsigned char *) malloc(_num_cols);
		bcopy(b.matrix[i], matrix[i], _num_cols);
	}
#endif
}

// **** generate ****
// This function generates a new random combination from the packets stored in the buffer. The coefficients are randomly chosen in a finite field. 
// ******************

Packet* NCBuffer::generate() {
#ifdef NC_DEBUG
	printf("NCBuffer:generate() \n"); 
#endif
	assert(generated_rank_ <= _rank);
#ifdef NC_DEBUG
	printf("Generated rank %i generated %i \n", generated_rank_, generated_);
#endif	
	if (generated_ && generated_rank_ == _rank)
		return generated_;

#ifdef USE_NTL	
	vec_GF2E res;
	res.SetLength(_num_cols);
	GF2E r;
#else
	unsigned char* res = (unsigned char *) malloc(_num_cols);
	bzero(res, _num_cols);
	unsigned char r;
#endif
	/* create random permutation of nodes */
        int *perm = new int[block_size];
        permutation(perm, block_size, NC_rngB);

	int count = 0;
	for(int i = 0; i < block_size; ++i) { 
		if (genvec_maxentries == 0 || 
		    (genvec_maxentries < 1.0 && NC_rngB->genrand_real() <= genvec_maxentries) ||
		    (genvec_maxentries >= 1.0 && count < genvec_maxentries)) {
			    do {
#ifdef USE_NTL	
				    r = random_GF2E();
#else
				    r = (unsigned char) NC_rngB->genrand_int32();
#endif
			    } while (r == 0);
#ifdef USE_NTL
			    res += r * matrix[perm[i]];
#else
			    for(int j = 0; j < _num_cols; ++j)
				    res[j] ^= GF2E::mul(r, matrix[perm[i]][j]);
#endif
			    count = 0;
			    for(int j = 0; j < _num_cols; ++j)
				    if (res[j] != 0)
					    ++count;
		}
	}

	delete[] perm;

#ifdef DEBUG_OUTPUT
	cout << "block_size " << block_size << " - generated " << res << " using " << rv << endl;
#endif
		
	
 
	generated_ = Packet::alloc();
	
	struct hdr_cmn *ch = HDR_CMN(generated_);
	struct hdr_ip *ih = HDR_IP(generated_);
	struct hdr_NC *nh = hdr_NC::access(generated_);
	
	//NC header
	nh->allocCoefficient(_num_cols);
	memcpy(nh->coefficient(), res, _num_cols);
	free(res);
	nh->generation() = _gen;
	nh->lenCoeff() = _num_cols;
	nh->length() = _num_cols;
	
	//Common header
	ch->size() = NC_HDR_LEN + _num_cols;
	generated_rank_ = _rank;
	return generated_;
}


bool NCBuffer::fits(Packet *d) {
#ifdef NC_DEBUG
	printf("NCBuffer:fits() \n"); 
#endif

	struct hdr_NC *nh = hdr_NC::access(d);
 	return (nh->generation() == _gen);
}


// **** inject ****
// This function insters a packets in the buffer and the coefficient in the coding matrix. After each injection the matrix is reduced in the Gauss form 
// **************** 

void NCBuffer::inject(Packet *d) {
#ifdef NC_DEBUG_final
	printf("NCBuffer:inject(), block_size = % i \n", block_size); 
#endif

	assert(fits(d));
	struct hdr_NC *nh = hdr_NC::access(d);

#ifdef USE_NTL	
	assert(nh->lenCoeff() == _num_cols);
#else
	assert(nh->lenCoeff() == _num_cols);
#endif

#ifdef NC_DEBUG_final
	printf(" Coefficient of the injected packet, block_size = %i lencoeff %i \n", block_size, nh->lenCoeff());
	nh->printCoeff(block_size);
	printf(" rank %li\n", _rank);
#endif

	_innovative = false;
	if (_rank < block_size) {
		int oldrank = _rank;

	//Matrix is always stored echelonated, so it has 'rank' non-null rows	
#ifdef USE_NTL	
		matrix[block_size-1] = nh->coefficient();
		_rank = gauss(matrix);
#else
		memcpy(matrix[block_size-1], nh->coefficient(), nh->lenCoeff());

#ifdef NC_DEBUG_final
	printf("Matrix before gaussian reduction \n", _innovative);	
	for (int j = 0; j < block_size; j++) {
		for (int i = 0; i < block_size; i++)
			printf("%2x ", matrix[j][i]);
		printf(" --------\n");
	}
	
#endif
		_rank = GF2E::gauss(matrix, block_size, nh->lenCoeff());
#endif

		if (_rank > oldrank) {
			full_invert();
			_innovative = true;

			// recalc # of used columns
			_used_cols = block_size;
			for (int i = 0; i < block_size; i++) {
				int j = 0;
				while (j < block_size && matrix[j][i] == 0)
					++j;
				if (j == block_size)
					--_used_cols;
			}
			
			// Evaluate if the node has decoded all it can
			
			full_decoded = false;
			decoded_pkt = 0;
			for (int i=0; i < block_size; i++){
				if (_decoded[i] > 0)
					decoded_pkt++;
			}
			
			if (decoded_pkt == _rank)
				full_decoded = true;

		}
	}


#ifdef NC_DEBUG_final
printf("Matrix after gaussian reduction \n", _innovative);
	for (int j = 0; j < block_size; j++) {
		for (int i = 0; i < _num_cols; i++)
			printf("%2x ", matrix[j][i]);
		printf(" --------\n");
	}
	printf("The packet is innovative = %i \n", _innovative);
#endif
	
#ifdef DEBUG_OUTPUT
	cout << "inserted " << ((_innovative) ? "":"non-") << "innovative packet:" << d->contents << endl;
	cout << matrix << endl;
#endif
	
	return;
}


//**** priority ****
// This function returns the priority of the buffer according to different strategies
// n = 0: Normal Priority based on the sendallowed and sendcount counters (p = sendallowed - sendcount)
// n = 1: Stopping priority based on the situation of node's neighbors estimated on the fly.
// ******************

double NCBuffer::priority(int n) {
#ifdef NC_DEBUG_final
	printf("NCBuffer:priority(), _sendcount = %i, _sendallowed %i \n", _sendcount, _sendallowed); 
#endif

	assert(_recvcount >= _rank);

	if (opt_fullrankfw && _rank < block_size)
		return 0;
	if (_sendcount >= abs_send_count)
		return 0;
	
	// number of packets the node is allowed to send out
	double val;
	if(n == 0){ 
		val = (double)_sendallowed - (double)_sendcount;
		// "- _rank" to account for innovative packets
		if (recv_count > 0)
			val -= (_recvcount - _rank) / recv_count;
	} else if (n == 1){
		int count_n = 0;
		int count_s = 0;
		int rank_tmp = 0;
		bool first = true;
		for(int i = 0; i < 200; i++){
			if(stop_vector[i] > -1){ 
				++count_n;
				if(stop_vector[i] > 0 && first == true){
					rank_tmp = stop_vector[i];
					first = false;
				}
			}
			if (stop_vector[i] > 0 && stop_vector[i] == rank_tmp && first == false){
					++count_s;
			}
		}
		if (count_n == 0)
			val = 1;
		else
			val = count_n - count_s;
			//printf("Val = %f - neigh = %i - count_n = %i - count_s = %i \n", val, neigh, count_n, count_s);
	}
	return val;
}

void NCBuffer::set_stop_vector(int n){

	for(int i = 0; i < 200; i++)
		stop_vector[i] = -1;
}

bool NCBuffer::set_new_neigh(int n){
	bool new_n = false;
		if(stop_vector[n] == -1){
			stop_vector[n] = 0;
			new_n = true;
		}
	return new_n;
}

bool NCBuffer::disposable() {
	// XXX not implemented
	assert(0);
	return 0;
}

void NCBuffer::dump() {
	printf("[Gen: %i, Rank: %i, SndCnt: %i, RecvCnt: %i, Priority: %f ] \n", _gen, _rank, _sendcount, _recvcount, priority(0));
}


// **** full_invert ****
// This function is used to invert the Gauss matrix and so decode the packets
// *********************

void NCBuffer::full_invert() {
#ifdef NC_DEBUG
	printf("NCBuffer:full_invert() \n"); 
#endif


	int row, s, t;

	// this assumes gauss() was run on matrix before
	for (row = 0, s = 0; row < block_size; ++row) {
		// find pivot
		while (s < block_size && matrix[row][s] == 0)
			++s;
		if (s == block_size)
			break;

		if (matrix[row][s] != 1) {
#ifdef USE_NTL	
			matrix[row] *= inv(matrix[row][s]);
#else
			for(int j = 0; j < _num_cols; ++j)
				matrix[row][j] = GF2E::mul(matrix[row][j], GF2E::inv(matrix[row][s]));
#endif
		}
		assert(matrix[row][s] == 1);
		for (t = row - 1; t >= 0; t--) {
			if (matrix[t][s] != 0) {
#ifdef USE_NTL	
				matrix[t] -= (matrix[row] * matrix[t][s]);
#else
				unsigned char mult = matrix[t][s];
				for(int j = 0; j < _num_cols; ++j)
					matrix[t][j] ^= GF2E::mul(matrix[row][j], mult);
#endif
			}
			assert(matrix[t][s] == 0);
		}
	}

	// update decodable columns
	for (row = 0, s = 0; row < block_size; ++row) {
		// find pivot
		while (s < block_size && matrix[row][s] == 0)
			++s;
		if (s == block_size)
			break;
		assert(matrix[row][s] == 1);
		for (t = s + 1; t < block_size && matrix[row][t] == 0; ++t)
			;
		if (t == block_size)
			++_decoded[s];
	}

}


unsigned char* NCBuffer::recover(int col) {
#ifdef NC_DEBUG
	printf("NCBuffer:recover() \n"); 
#endif

	if (!_decoded[col] || _num_cols == block_size)
		return 0;

	// find matrix row
	int row;
	for (row = 0; row < block_size; ++row) {
		if (matrix[row][col] == 1) {
#ifdef DEBUG			
			for (int s = 0; s < block_size; ++s) {
				if (s != col) {
					assert(matrix[row][s] == 0);
				}
			}
#endif
			break;
		}
	}
	assert(row < block_size);

	assert(field_size > 0);
	// convert buffer content
	int s = _num_cols - block_size;
	unsigned char *dest = new unsigned char[s];
	assert(dest);
#ifdef USE_NTL	
	unsigned char *d = dest;
	for (int i = block_size; i < block_size+s; i++) {
		BytesFromGF2X(d++, rep(matrix[row][i]), 1);
	}
#else
	memcpy(dest, &matrix[row][block_size], s);
#endif
	return dest;
}	



void NCBuffer::permutation(int *array, int size, RNG_NC *rng) {
#ifdef NC_DEBUG
	printf("NCBuffer:permutation() \n"); 
#endif

	/* random permutation of index array */
	int tmp, j;
	for (int i = 0; i < size; ++i)
		array[i] = i;

	for (int i = size; i > 0; --i) {
		j = rng->genrand_int32() % i;
		tmp = array[i-1];
		array[i-1] = array[j];
		array[j] = tmp;
	}
}	

NCBufferAllocation::NCBufferAllocation() : genmax(-1), genlast(-1) {}

void NCBufferAllocation::init(long s) {
#ifdef NC_DEBUG
	printf("NCBufferAllocation:init() \n"); 
#endif
	// create allocation table for all app generated packets
	// (need to do this after constructor to make sure the parameters are already parsed)
	//block = b;
	list_size = s;
	alloc_list = new Packet*[s];
	gen_origin = new int[s]; // (Note: s could be divided by avg matrix size)
	for (int i = 0; i < s; i++)
		alloc_list[i] = 0;
}

int NCBufferAllocation::num_columns(long gen) {
#ifdef NC_DEBUG
	printf("NCBufferAllocation:num_columns() \n"); 
#endif
	// return largest used column
	assert(gen <= genmax && gen >= 0);
	int offset = gen * NCBuffer::block_size;
	for (int col = NCBuffer::block_size; col > 0; --col) {
		if (alloc_list[offset + col - 1])
			return col;
	}
	return 0;
}

Packet* NCBufferAllocation::get_alloc(long gen, long col) {
#ifdef NC_DEBUG
	printf("NCBufferAllocation:get_alloc() \n"); 
#endif
	assert(gen * NCBuffer::block_size + col < list_size);
	assert(gen <= genmax && col < NCBuffer::block_size);
#ifdef NC_DEBUG_final
		if(alloc_list[gen * NCBuffer::block_size + col]){
			struct hdr_ip *ih = HDR_IP(alloc_list[gen * NCBuffer::block_size + col]);
			printf("ATTENZIONE: get alloc position %i Packet saddr %i \n", gen * 	NCBuffer::block_size + col, ih->saddr());
		}	
#endif
	return alloc_list[gen * NCBuffer::block_size + col];
}

int NCBufferAllocation::alloc_insert(Packet *p, int nid, int gen) {
#ifdef NC_DEBUG
	printf("NCBufferAllocation:alloc_insert(): "); 
#endif
	int col = 0, g = gen * NCBuffer::block_size;
	assert(gen >= 0);

	// find suitable column
	while (alloc_list[g + col] && col < NCBuffer::block_size)
		++col;

	// col == block -> no empty column found in gen
	if (col < NCBuffer::block_size) {
		if (col == 0)
			gen_origin[gen] = nid;

		genlast = gen;
		if (gen > genmax)
			genmax = gen;

		// XXX ugly: initial packets are never deleted, same for the
		// arrays of packet pointers
		assert(g + col < list_size);
		alloc_list[g + col] = p->copy();
#ifdef NC_DEBUG_final
		struct hdr_cmn *ch = HDR_CMN(alloc_list[g + col]);
		struct hdr_ip *ih = HDR_IP(alloc_list[g + col]);
		struct hdr_NC *nh = hdr_NC::access(alloc_list[g + col]);
		printf("\t FIELD of the inserted packet: saddr %i, daddr %i, uid %i, size %i, ptype %i \n", ih->saddr(), ih->daddr(), ch->uid(), ch->size(), ch->ptype());
#endif
	}
	return col;
}


