/*
 * Copyright (c) 2009 Regents of the SIGNET lab, University of Padova and DOCOMO Communications Laboratories Europe GmbH.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Padova (SIGNET lab) nor the 
 *    names of its contributors may be used to endorse or promote products 
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED 
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef _NC_BUFFER_H
#define _NC_BUFFER_H

#ifdef USE_NTL
#include <NTL/vec_GF2E.h>
#include <NTL/mat_GF2E.h>
#include <NTL/GF2XFactoring.h>
using namespace NTL;
#else
#include "gf2e.h"
#endif
#include "rng_NC.h"

#include <agent.h>
#include <packet.h>
#include <sys/types.h>
#include <scheduler.h>
#include <cmu-trace.h>
#include <priqueue.h>
#include <classifier/classifier-port.h>
#include <packet.h>



using namespace std;

/** 
* This class contains all the structures and methods to store packets within a node and support network coding functionalities 
*/

class NCBuffer {

	
 public:
	// send & receive counter
	double _sendallowed;
	int _sendcount;
	int _recvcount;
	/// \brief Generation of this buffer 
	long _gen; 		
	/// \brief New random combination
	Packet *generated_;	
	/// \brief Rank at last generation of a new combination
	long generated_rank_;	

	// Content matrix and rank at last gaussian reduction

#ifdef USE_NTL	
	mat_GF2E matrix;
#else
	unsigned char **matrix;
#endif
	long _rank;
	int _num_cols;
	int _used_cols;

	/// \brief There is at least one innovative packet	
	bool _innovative;	
	/// \brief Rows that can already be decoded 
	int *_decoded; 		
	/// \brief Number of decoded packets	
	int decoded_pkt;
	/// \brief true if the node has decoded all it can	
	bool full_decoded;
	
	
	RNG_NC* NC_rngB;
	/** Operate a full inversion. Matrix HAS to be gauss-reducted BEFORE */
	void full_invert();
	int stop_vector[200]; 
	int already_notify;
	// Initialize an empty buffer
	/// \brief Contructors and Initialization
	NCBuffer(long g, int d, long int b); 
	NCBuffer(const NCBuffer &b);
	~NCBuffer() {
#ifndef USE_NTL	
		for (int i = 0; i < block_size; ++i)
			free(matrix[i]);
#endif
	} 

	/** Initialize NTL with appropriate Galois Field, and set Block Size */
	static void init(int long b);
	static int abs_send_count;

	/** Generate a randomly-coded packet
	 * from the buffer, and increment sendcount. Datagrams
	 * have to be freed by the caller once no longer in use */
	Packet *generate();
	Packet *getGenerated() { return generated_; }
	void reset_generated() { generated_ = 0; generated_rank_ = 0; }

	/** Checks if a datagram can fit into a buffer (i.e, has the
	 * right generation */
	bool fits(Packet *d);	

	/** Inject a received datagram into the buffer 
	 * and sets _innovative accordingly */
	void inject(Packet *d);	

	/** Rank  */
	long rank() { return _rank; }
	/** Innovative packet  */
	bool innovative() { return _innovative; }
	/** Decoded packets  */
	int decoded(int r) { return _decoded[r]; }
	/** The matrix is full decodable or not  */
	bool is_full_decoded() { return full_decoded; }

	/** Generation of the buffer */
	long generation() { return _gen; }
	/** This function returns the priority of the buffer according to different strategies
	* n = 0: Normal Priority based on the sendallowed and sendcount counters (p = sendallowed - sendcount)
	* n = 1: Stopping priority based on the situation of node's neighbors estimated on the fly.
	*/
	double priority(int n);
	bool disposable();	

	double sendallowed() {return _sendallowed; }
	int recvcount() { return _recvcount; } 
	int sendcount() { return _sendcount; }
	void inc_sendallowed(double inc) { _sendallowed += inc; }
	void inc_recvcount() { ++_recvcount; }
	void inc_sendcount() { ++_sendcount; }
	void permutation(int *array, int size, RNG_NC *rng);
	void set_stop_vector(int n); 
	void update_stop_vector(int n, long int r){
		stop_vector[n] = r;
	}
	bool set_new_neigh(int n);
	int get_notify(){return already_notify;} 
	void inc_notify(){
		++already_notify;
	} 
	

	/** Estimated hopcount */
	int hopcount();

	/** Decode the buffer contents and put it into a byte array. */
	unsigned char* recover(int idx);

	/** Size of the Galois Field */
	static int field_size;
	/** Dimensions of the space */
	static int block_size;
	static int target_size;

	static int opt_fullrankfw;

	/* if genvec_maxentries>1 : try to keep number of non-zero entries below genvec_maxentries */
	/* if genvec_maxentries<1 : fraction of 0 in random vector with which we multiply */
	static double genvec_maxentries;

	/* Prints the buffer status on stdout */
	void dump(); //ostream &out
	
	static int recv_count;
};

/**
* This class manages the generation of new packets
*/

class NCBufferAllocation {
	
 private:
	/** Which original packet is allocated to which generation and column */
	Packet **alloc_list;
	/** id of node that created generation */
	int *gen_origin;

	long list_size;

	/** Global generation counter (global highest generation ID) */
	long genmax;
	/** Last generation that was used */
	long genlast;

 public:

	NCBufferAllocation();
	void init(long s);

	int alloc_insert(Packet *p, int nid, int gen);
	Packet* get_alloc(long gen, long column);
	int get_origin(long g) { return gen_origin[g]; }
	int num_columns(long gen);
	long max_generation() { return genmax; }
	long last_generation() { return genlast; }
	Packet* get_packet(int i) {return alloc_list[i];}
};

#endif
