function [x, y, xt, yD, Ev, SDt, NoiseNorms, t] = BuildData(y0, t, ghQR_fun)
% BuildData - Simulates a state-space model with process and measurement noise.
%
% INPUTS:
%   y0         - Initial condition (column vector)
%   t          - Time vector
%   ghQR_fun   - Function handle to the function returning g, h, Q, R
%
% OUTPUTS:
%   xt         - True states (noise-free)
%   x          - Observed states (noisy input for regression)
%   y          - Output for regression (shifted first state)
%   yD         - Decomposition of the signal y (deterministic, process, measurement)
%   Ev         - Measurement noise covariance matrix
%   SDprof     - Profile of the process-noise standard deviations for the first equation
% NoiseNorms   -  Measurement noise / process noise
% t            - sampling times

Num = length(t);
nx = length(y0); % state dimension

NoiseNorms = inf;      % Initialize ratio norm (measurement noise / process noise)
Sl=0.4;%threshold
% Regenerate data until the ratio of noise norms is below a threshold
while NoiseNorms > Sl
    
    % Matrix initialization
    G = zeros(nx, Num);
    Nt = zeros(nx, Num);
    Xt = zeros(nx, Num);
    N = zeros(nx, Num);
    Xn = zeros(nx, Num);

    % Initial call to the provided function
    CV=[0.04*rand*ones(1,2) 1]; % noise level
    [g, h, Q, R] = ghQR_fun(y0,CV,t);
    yD=[];

    for i = 1:Num
        % Deterministic state
        G(:, i) = g;

        % Process noise
        Nt(:, i) = sqrtm(Q) * randn(nx, 1);

        % True state
        Xt(:, i) = G(:, i) + Nt(:, i);
        Xt(:, i)  = abs(Xt(:, i));  % protection from negativity

        % Noisy observation
        [g, h, Q, R] = ghQR_fun(Xt(:, i),CV,t);

        N(:, i) = sqrtm(R) * randn(nx, 1);
        Xn(:, i) = h + N(:, i);
        Xn(:, i)  = abs(Xn(:, i));  % protection from negativity
    end

    % Data for regression
    xt = Xt(:, 1:end-1)'; % Noise-free input
    x  = Xn(:, 1:end-1)'; % Noisy input
    y  = Xn(1, 2:end);    % Output: first state at next step

    % Signal decomposition for y
    yD(1, :) = G(1, 2:end);  % deterministic component
    yD(2, :) = Nt(1, 2:end); % process noise
    yD(3, :) = N(1, 2:end);  % measurement noise

    % Measurement noise covariance
    if CV(end) == 1
        Ev = diag((CV(1) * y).^2);
    else
        Ev = (CV(1)^2) * eye(length(y));
    end

    % Standard deviation profile for the first state
    for i = 1:size(x, 1)
        [~, ~, Q, ~] = ghQR_fun(x(i, :)',CV,t);
        SDt(i) = sqrt(Q(1, 1));
    end

    % Downsample every 10 points
    sel = 10;
    yD = yD(:,sel:sel:end);
    NoiseNorms = [norm(yD(3,:)) / norm(yD(2,:))]; % Measurement noise / process noise
    if NoiseNorms<=Sl%training data definition
        t = t(sel:sel:end);
        x = x(sel:sel:end,:);
        y = y(sel:sel:end);
        xt = xt(sel:sel:end,:);
        v = diag(Ev);
        Ev = diag(v(sel:sel:end));
        SDt = SDt(sel:sel:end);
    end

end

end
