function [g,h,Q,R] = crea_ghQR_2D(n, CV, dt)
% Toggle Switch: function for a two-dimensional model (x1, x2) with mutual repressors

%%
dt = dt(1);             % extract scalar time step
theta  = 1e4;           % active production [molecules/s]
alpha  = 1000;          % degradation rate [1/s]
beta   = 50;            % cooperativity
delta  = 0.75;          % switch ON rate [1/s]

%% 
m0 = 1000;              % molecular scaling factor

alpha1 = alpha;         % rescaled degradation rate
%delta1 = delta;        % (unused) alternative scaling

K = 0.01;               % repression strength parameter
                        % Larger K → stronger separation between states and clearer bimodality
                        % Smaller K → stronger noise and overlapping modes

%b = 0.5;               % manually tuned value (alternative)
b = beta*theta*(delta^2)/(alpha1^2);    % expression for cooperativity scaling
kappa = K*(alpha1^2)/(theta*(delta^3)); % non-dimensional parameter

%%
x1 = n(1) / m0;         % normalized state 1
x2 = n(2) / m0;         % normalized state 2

%%
[A1, A2, B1, B2, B12] = mutual_repressors(x1, x2, b, kappa, m0);  % compute drift and diffusion terms

%%
g = n + dt * m0 * [A1; A2];    % drift term (deterministic update)

%%
Q = dt * m0^2 * [B1, B12; B12, B2];    % process noise covariance

%%
h = n;                     % measurement function (identity)

%%
if CV(3) == 1
    R = diag((CV(1) * n).^2);  % proportional measurement noise
else
    R = (CV(1)^2) * eye(2);    % constant measurement noise
end
end

%%
function [A1, A2, B1, B2, B12] = mutual_repressors(x1, x2, b, kappa, m0)
    % Mutual repression model functions:
    % A1, A2 = drift terms
    % B1, B2 = diffusion terms
    % B12    = cross-correlation term

    % Hill-type repression fractions
    frac1 = 1 / (1 + (x2^2)/(b + x1^2));
    frac2 = 1 / (1 + (x1^2)/(b + x2^2));

    % Common factors used in the stochastic contributions
    common_denom = b * (b + x1^2 + x2^2)^4;
    common_factor = 2 * x1 * x2 * (x1 + x2);

    % Polynomial expressions for the drift corrections
    poly1 = (x1 - 1)*(b + x1^2)*(2*b + x1^2) ...
          + x1*(x2^2)*(3*b + x1*(2*x1 - 1)) + x1*(x2^4);

    poly2 = (x2 - 1)*(b + x2^2)*(2*b + x2^2) ...
          + x2*(x1^2)*(3*b + x2*(2*x2 - 1)) + x2*(x1^4);

    % Drift equations A1, A2
    A1 = frac1 - x1 - (1/kappa)*(common_factor * poly1 / common_denom);
    A2 = frac2 - x2 - (1/kappa)*(common_factor * poly2 / common_denom);

    % Deterministic part of diffusion (intrinsic noise)
    B1_det = (1/m0) * (frac1 + x1);
    B2_det = (1/m0) * (frac2 + x2);

    % Stochastic part of diffusion (extrinsic noise)
    B1_stoch = (1/kappa) * ((x2^2) * (b^2 + 2*b*(x1^2) + (x1^2)*(x2^2) + x1^4)) / (b*(b + x1^2 + x2^2)^3);
    B2_stoch = (1/kappa) * ((x1^2) * (b^2 + 2*b*(x2^2) + (x1^2)*(x2^2) + x2^4)) / (b*(b + x1^2 + x2^2)^3);
    B12 = (1/kappa) * ((x1^2) * (x2^2) * (2*b + x1^2 + x2^2)) / (b*(b + x1^2 + x2^2)^3);

    % Total diffusion terms
    B1 = B1_det + B1_stoch;
    B2 = B2_det + B2_stoch;
end





















































