
clear all
close all
warning off
global Ev S Kw NumX EvIN

MC=2000; % number of Monte Carlo runs

for k=1:MC;k

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%% DATA GENERATION %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
ghQR_fun = @crea_ghQR_2D; % Model-specific function returning g, h, Q, R
BiasSpace = 1;            % If 1: add linear basis to GPR model; if 0: no linear basis
NumX = [1:2];             % State components used as input locations for transition noise kernel

y0 = 100*[0.5; 0.1];  % Initial condition for the 2D state
t = [0.01:0.01:100];  % Time vector

% Generate synthetic data using state-space simulator
[x,y,xt,yD,Ev,SDt,NoiseNorms(k),t] = BuildData(y0,t,ghQR_fun);
% True intrinsic noise realizations 
Rest = yD(2,:);
%training data (x,y) are now available
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%% TRINE: FIRST STAGE %%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Kernel for deterministic part + white noise
GK1 = @(XN,XM,theta)((exp(theta(2))^2)*exp(-(pdist2(XN,XM).^2)/(2*exp(theta(1))^2)) + Ev);
theta1 = [0.1, 10]; % Initial kernel hyperparameters

if BiasSpace==0
    % GPR with constant basis (default)
    gprMdl1 = fitrgp(x,y,'KernelFunction',GK1,'KernelParameters',theta1);
elseif BiasSpace==1
    % GPR with linear basis functions
    gprMdl1 = fitrgp(x,y,'KernelFunction',GK1,'KernelParameters',theta1,'BasisFunction','Linear');
end

ypred1 = resubPredict(gprMdl1)';    % Estimated deterministic + measurement noise
Res1 = y - ypred1;                  % Estimated process noise (stage 1)
S = diag(sign(Res1));               % Sign matrix for Trine kernel

% Compute predictive variance of GPR model
theta1 = gprMdl1.KernelInformation.KernelParameters;
K = GK1(x,x,theta1);
n = size(x,1);
Ky = K + (gprMdl1.Sigma^2)*eye(n);
L = chol(Ky,'lower'); % Ky = L*L'
V = L \ K;
yvar = diag(K - V'*V) + gprMdl1.Sigma^2;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%% TRINE: SECOND STAGE %%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Kw models multiplicative white noise interaction
Kw = (2/pi)*ones(length(y),length(y)) + (1-2/pi)*eye(length(y));
% Full TRINE kernel: deterministic + measurement noise + structured process noise
GK2 = @(XN,XM,theta)(...
    (exp(theta(2))^2)*exp(-(pdist2(XN,XM).^2)/(2*exp(theta(1))^2)) + ... % deterministic
    Ev + ...                                                            % measurement noise
    S*[ (exp(theta(4))^2)*exp(-(pdist2(XN(:,NumX),XM(:,NumX)).^2)/(2*exp(theta(3))^2)) .* Kw ]*S');

theta2 = [0.1,10,0.1,10];
Sigma = 1e-6; % small noise SD: GK2 contains full variance

if BiasSpace==0
    gprMdl2 = fitrgp(x,y,'KernelFunction',GK2,'KernelParameters',theta2,'Sigma',2*Sigma,'ConstantSigma',true,'SigmaLowerBound',Sigma);
elseif BiasSpace==1
    gprMdl2 = fitrgp(x,y,'KernelFunction',GK2,'KernelParameters',theta2,...
        'Sigma',2*Sigma,'ConstantSigma',true,'SigmaLowerBound',Sigma,'BasisFunction','Linear');
end
theta2 = gprMdl2.KernelInformation.KernelParameters;
% Compute structured covariance C for intrinsic noise
C = S*[ (exp(theta2(4))^2)*exp(-(pdist2(x(:,NumX),x(:,NumX)).^2)/(2*exp(theta2(3))^2)) .* Kw ]*S';
% Full covariance of y
Ey = (exp(theta2(2))^2)*exp(-(pdist2(x,x).^2)/(2*exp(theta2(1))^2)) + Ev + C;
% Compute estimated intrinsic-noise realization (stage 2)
if BiasSpace==0
    Res2 = C*inv(Ey)*(y - gprMdl2.Beta)';
elseif BiasSpace==1
    if length(y0)==2
        Res2 = C*inv(Ey)*(y - gprMdl2.Beta(1) - gprMdl2.Beta(2)*x(:,1)' - gprMdl2.Beta(3)*x(:,2)')';
    elseif length(y0)==1
        Res2 = C*inv(Ey)*(y - gprMdl2.Beta(1) - gprMdl2.Beta(2)*x(:,1)')';
    end
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%% TRINE: THIRD STAGE %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Regression of |noise| to estimate SD profile
gprMdl3 = fitrgp(x,sqrt(pi/2)*abs(Res2),'BasisFunction','none');
Respred3 = resubPredict(gprMdl3)';%the estimated SD profile from Trine
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% TRINE^u %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
gprMdl4 = fitrgp(x,sqrt(pi/2)*abs(Res1),'BasisFunction','None');
Respred4 = resubPredict(gprMdl4)';%the estimated SD profile from Trine^u
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% ORACLE %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
gprMdl5 = fitrgp(x,sqrt(pi/2)*abs(Rest),'BasisFunction','None');
Respred5 = resubPredict(gprMdl5)';%the estimated SD profile from Oracle
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% MLHGP %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
NiterMLH = 10; % inner MLHGP iterations
yn = log(0.5*(Res1.^2 + yvar'));
gprMdl7 = fitrgp(x,yn,'BasisFunction','None');
Respred7 = sqrt(exp(resubPredict(gprMdl7)'));%the estimated SD profile from MLHGP, then refined using MLHGP iterations
FitMLH(k,1) = 100*(1 - norm(Respred7-SDt)/norm(SDt));
%%%%%%% MLHGP ITERATIONS %%%%%%%
for hh=2:NiterMLH;hh
    EvIN = diag(Respred7.^2); % Iteratively updated noise covariance
    GK3 = @(XN,XM,theta)(...
        (exp(theta(2))^2)*exp(-(pdist2(XN,XM).^2)/(2*exp(theta(1))^2)) + ...
        Ev + EvIN );
    theta1 = [0.1,10];
    Sigma = 1e-6;
    if BiasSpace==0
        gprMdlMLH = fitrgp(x,y,'KernelFunction',GK3,'KernelParameters',theta2,'Sigma',2*Sigma,'ConstantSigma',true,'SigmaLowerBound',Sigma);
    elseif BiasSpace==1
        gprMdlMLH = fitrgp(x,y,'KernelFunction',GK3,'KernelParameters',theta2,...
            'Sigma',2*Sigma,'ConstantSigma',true,'SigmaLowerBound',Sigma,'BasisFunction','Linear');
    end
    thetaMLH = gprMdlMLH.KernelInformation.KernelParameters;
    K12 = (exp(thetaMLH(2))^2)*exp(-(pdist2(x,x).^2)/(2*exp(thetaMLH(1))^2)) + Ev;
    Ey = K12 + EvIN;
    % Estimate intrinsic noise (IN), ResMLH will replace Res1
    if BiasSpace==0
        ResMLH = y - (K12*inv(Ey)*(y - gprMdlMLH.Beta)' + gprMdlMLH.Beta)';
    elseif BiasSpace==1
        if length(y0)==2
            ResMLH = y - (K12*inv(Ey)*(y-gprMdlMLH.Beta(1)-gprMdlMLH.Beta(2)*x(:,1)'-gprMdlMLH.Beta(3)*x(:,2)')' + gprMdlMLH.Beta(1) + gprMdlMLH.Beta(2)*x(:,1) + gprMdlMLH.Beta(3)*x(:,2))';
        elseif length(y0)==1
            ResMLH = y - (K12*inv(Ey)*(y-gprMdlMLH.Beta(1)-gprMdlMLH.Beta(2)*x(:,1)')' + gprMdlMLH.Beta(1) + gprMdlMLH.Beta(2)*x(:,1))';
        end
    end
    % Update predictive variance
    L = chol(Ky,'lower');
    V = L \ K12;
    yvar = diag(K12 - V'*V) + diag(EvIN);
    % Update target variable and re-train regression of SD
    yn = log(0.5*(ResMLH.^2 + yvar'));
    gprMdl7 = fitrgp(x,yn,'BasisFunction','None');
    Respred7 = sqrt(exp(resubPredict(gprMdl7)'));%updated estimate of SD profile from MLHGP
    FitMLH(k,hh) = 100*(1 - norm(Respred7-SDt)/norm(SDt));
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% RESULTS %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Fits of SD profile: Oracle, Trine, Trine^u, MLHGP')

FIT(k,:) = [ ...
    100*(1-norm(Respred5-SDt)/norm(SDt)), ... % Oracle
    100*(1-norm(Respred3-SDt)/norm(SDt)), ... % Trine
    100*(1-norm(Respred4-SDt)/norm(SDt)), ... % Trine^u
    100*(1-norm(Respred7-SDt)/norm(SDt))  ... % MLHGP
];
mean(FIT)

% Group results by noise-to-signal ratio
Sel = [0 0.1; 0.1 0.2; 0.2 0.3; 0.3 0.4];
for i=1:size(Sel,1)
    q = find(NoiseNorms > Sel(i,1) & NoiseNorms < Sel(i,2));
    Rip(i,:) = [mean(FIT(q,1)) mean(FIT(q,2)) mean(FIT(q,3)) mean(FIT(q,4)) length(q)];
end

clc; format short g
disp('Fits of SD profile: Oracle, Trine, Trine^u, MLHGP')
Rip

if k>1
labels = {'Oracle', 'Trine', 'Trine^u', 'MLHGP'};
boxplot(FIT, 'Labels', labels)
end

end
