#ifndef _ARRAY_H_
#define _ARRAY_H_

#include <iostream.h>
#include "exception.h"
#include "strings.h"



template <class T>
class Array 
{
public:
    class ArrayException: public Exception {
    public:
      ArrayException(const String& aDescription )
        : Exception( aDescription ) {}
      ArrayException(const ArrayException& e )
        : Exception( e.description() ) {}
    };

    class IndexOutOfBoundsException: public ArrayException {
    public:
      IndexOutOfBoundsException(const String& aDescription )
        : ArrayException( aDescription ) {}
      IndexOutOfBoundsException(const IndexOutOfBoundsException& e )
        : ArrayException( e.description() ) {}
      virtual const String className() const 
      {
        return "IndexOutOfBoundsException";
      }
    };
    
    Array(int size=5);
    Array( const Array<T>& otherArray);
    const Array<T>& operator= ( const Array<T>& otherArray);
    ~Array();

    T& objectAt( int index ) const;
    T& operator[] ( int index ) const;
    int addObject( const T& newObject );
    void setObjectAt( const T& newObject, int index );
    void removeObjectAt( int index );
    void removeAll();
    int count() const;
    int size() const;
    void printOn( ostream& outputStream = cout) const;

    bool objectDefinedAt( int index ); // tigre*:
private:
  	T* theElements;
	bool* freePlace;		// vero se il posto i-esimo  libero
  	int theMaximumSize;
 	int theElementsCount;
  
  	void resize(int newSize);
};

// tigre*>
template <class T>
inline bool Array<T>::objectDefinedAt( int index )
{
	return !freePlace[index];
}
// <*tigre

template <class T>
inline Array<T>::Array( int size )
{
  theMaximumSize = size;

  if ( theMaximumSize >= 0 ) {
    theElements = new T[theMaximumSize];
    freePlace = new bool[theMaximumSize];
  }
  else {
    theElements = NULL;
    freePlace = NULL;
    throwException( ArrayException( "Arraysize less than 0" ) );
  }
  theElementsCount = 0;
}


template <class T>
inline Array<T>::Array( const Array<T>& otherArray )
{
  theMaximumSize  = otherArray.theMaximumSize;
  theElementsCount = otherArray.theElementsCount;

  if( theElementsCount >= 0 ) {
    theElements = new T[theMaximumSize];
    freePlace = new bool[theMaximumSize];
    for (int i = 0; i < theElementsCount; i++ ) {
      theElements[i] = otherArray.theElements[i];
	  freePlace[i] = otherArray.freePlace[i];	
    }
  }
  else {
    theElements = NULL;
    freePlace = NULL;
    throwException( ArrayException( "Arraysize less than 0" ) );
  }
}

  
template <class T>
inline const Array<T>& Array<T>::operator= ( const Array<T>& otherArray )
{
  if ( this == &otherArray ) {
    return *this;
  }
  
  theElementsCount = otherArray.theElementsCount;
  
  if ( otherArray.theMaximumSize > theMaximumSize ) {
    if( theElements!=NULL ) delete [] theElements; // tigre*:
    if( freePlace!=NULL ) delete [] freePlace; // tigre*:
    theMaximumSize = otherArray.theMaximumSize;
    theElements = new T[theMaximumSize];
    freePlace = new bool[theMaximumSize]; // tigre*:
  }
  for ( int i = 0; i < theElementsCount; i++ ) {
    theElements[i] = otherArray.theElements[i];
    freePlace[i] = otherArray.freePlace[i];	
  }
  
  return *this;
}

  
template <class T>
inline Array<T>::~Array()
{
  if( theElements ) {
    delete [] theElements;
  }
  if (freePlace)
	delete [] freePlace;
}


template <class T>
inline T& Array<T>::operator[] ( int index ) const
{
  return objectAt( index );
}


template <class T>
inline T& Array<T>::objectAt( int index ) const
{
  if (freePlace[index])
  	throwException( ArrayException("Element "+String(index)+" undefined"));
  		
  if( index < 0 || index >= theElementsCount ) {
    throwException( IndexOutOfBoundsException( 
        "Index out of bounds with index " + String( index ) +
        " ( array size " + String( theElementsCount ) + " )" ) );
  }
  return theElements[index];
}


template <class T>
inline int Array<T>::addObject( const T &newObject )
{
  for (int i=0; i < theElementsCount; ++i)
  	if (freePlace[i]) {						// se trova un posto libero
		theElements[i] = newObject;			// lo usa
		freePlace[i] = false;
		return i;
  	}
  			
  if( theElementsCount >= theMaximumSize ) { // altrimenti cerca uno spazio alla fine
    resize( theMaximumSize + 5 );			// se non c' allarga l'array
  }
  theElements[theElementsCount] = newObject; // assegna 
  freePlace[theElementsCount] = false; 	
  return theElementsCount++;
}

template <class T>
inline void Array<T>::setObjectAt( const T& newObject, int index )
{
  if( index < 0 || index >= theElementsCount ) {
    throwException( IndexOutOfBoundsException( "Index out of bounds" ) );
  }
  theElements[index] = newObject;
  freePlace[index] = false;
}


template <class T>
inline void Array<T>::removeObjectAt( int index )
{
  if(index < 0 || index >= theElementsCount ) {
    throwException( IndexOutOfBoundsException( "Index out of bounds" ) );
  }

  //freePlace[index] = true;
  //theElementsCount--;

	// tigre*>
	theElements[index] = theElements[theElementsCount-1];
	freePlace[theElementsCount-1] = true;
	theElementsCount--;
	// <*tigre
}


template <class T>
inline void Array<T>::removeAll()
{
  theElementsCount = 0;
}


template <class T>
inline int Array<T>::count() const
{
  return theElementsCount;
}

template <class T>
inline int Array<T>::size() const
{
  return theMaximumSize;
}

template <class T>
inline void Array<T>::resize( int newSize )
{
  if( newSize < 0 ) {
    throwException( ArrayException( "Arraysize less than 0" ) );
  }
  T *newElements = new T[newSize];
  bool* newFreePlace = new bool[newSize];
  for( int i = 0; i < theElementsCount; i++ ) {
    newElements[i] = theElements[i];
	newFreePlace[i] = freePlace[i];
  }
  if( theElements ) {
    delete [] theElements;
    delete [] freePlace;
  }
  theElements = newElements;
  freePlace = newFreePlace;
  theMaximumSize = newSize;
}

template <class T>
void Array<T>::printOn(ostream& outputStream) const
{
  outputStream << "( array with " << count() << " elements:";
  int i;
  for (i = 0; i < count(); i++) {
    outputStream  << endl << objectAt( i );
  }
  outputStream << " )" << endl;
}

template <class T>
inline ostream& operator << (ostream& outputStream, const Array<T>& array) 
{
  array.printOn( outputStream );
  return outputStream;
}



#endif


