#define BUFFER_SIZE 64000

#include "channel.h"

#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <stdio.h>

Channel::Channel() {
	active = false;
}

Channel::~Channel() {
	close();
}

int Channel::create(int port) {
	openSocket(SOCK_STREAM, port, BUFFER_SIZE);

// attiva la modalit di ascolto
	if (ownSocket >= FD_SETSIZE)
		throwException(Error("Troppi file descriptor aperti"));
	
	if (::listen(ownSocket, 1) < 0)
		throwException(Error("Non riesco ad ascoltare"));

	return ownPort;
}

void Channel::open(const String& hostName, int port) {
	openSocket(SOCK_STREAM, 0, BUFFER_SIZE);

	ioSocket = ownSocket; 	// il socket usato per la comunicazione  lo stesso di 
							// quello usato per la comunicazione (in modalit client)

// Effettua la connessione con il server
	sockaddr_in hostAddress = getHostInfo(hostName, port);
	if (connect(ownSocket, (sockaddr *) &hostAddress, sizeof(hostAddress)) < 0)
		throwException(Error("Connessione all'host " +hostName +  " port " + 
		      String((int) port) +	" fallita"));

	active = true;
}

void Channel::listen(int timeout) {
	fd_set fdSet;
	struct sockaddr_in clientAddress;
	int result;

	if (active == true)			// se c' una comunicazione sul socket di io 
		::close(ioSocket);		// lo chiude per permettere ad un altro di accedere
	
	FD_ZERO(&fdSet);
	FD_SET(ownSocket, &fdSet);

// Aspetta l'arrivo di una richiesta
	if (timeout == -1)			// se non c' un timeout
		result = select(FD_SETSIZE, &fdSet, NULL, NULL, NULL);
	else {						// altrimenti lo imposta
		struct timeval timer;
		timer.tv_sec = timeout;
		timer.tv_usec = 0;

		result = select(FD_SETSIZE, &fdSet, NULL, NULL, &timer);
	}
	
	if (result < 0) 
	;//	throwException(Error("Select fallito"));
	else
	if (result == 0)
		throw Timeout();

	size_t size = sizeof(clientAddress);
	ioSocket = accept(ownSocket, (sockaddr *) &clientAddress, &size);
	if (ioSocket < 0)
		throwException(Error("Non riesco a collegarmi al client"));

	active = true;
}

void Channel::close() {
	if (active == true)	// se il pipe  aperto lo chiude
		closeSocket();

	active = false;
}

void Channel::openSocket(int style, int port, size_t _bufferSize) {
	int socketID;
	struct sockaddr_in address;
	
// inizializza il  socket
	if (( socketID = socket(AF_INET, style, 0)) < 0) 
    	throwException(Error( "Inizializzazione del socket fallita") );

	if (( setsockopt (socketID, SOL_SOCKET, SO_RCVBUF,  (char*) &_bufferSize,
		    sizeof (int)) ) < 0) 
    	throwException(Error( "Inizializzazione del socket fallita") );
  

// fa il binding del socket
	memset( (char *) &address, 0, sizeof(address) );
  	address.sin_family = AF_INET;
  	address.sin_addr.s_addr = htonl(INADDR_ANY);
  	address.sin_port = htons(port);
  	if ( bind(socketID, (struct sockaddr *) &address, 
		    sizeof(address)) < 0) 
    	throwException( Error( "Binding of socket failed") );

// Recupera la porta
  	socklen_t socklen = sizeof(address);
  	if ( getsockname(socketID, (struct sockaddr *) &address, 
		    &socklen) < 0) 
    	throwException( Error( "Getting port of socket failed") );
  
	ownSocket = socketID;
	ownPort = ntohs(address.sin_port);

}

sockaddr_in Channel::getHostInfo(const String& hostName, int port) {
	hostent* hostEnvironment;
	in_addr* hostInternetAddress;
	sockaddr_in hostAddress;

// cerca l'host
	const char* host = hostName.cString();

	if (( hostEnvironment = (struct hostent*) gethostbyname(host)) == NULL) {
    if ( ((long) inet_addr(host)) == -1) {
      throwException( 
          Error( "Indirizzo dell'host non trovato") );
    }
	}
	else {
		hostInternetAddress = (struct in_addr*) *hostEnvironment->h_addr_list;
		host = inet_ntoa (*hostInternetAddress);
	}
	
	memset( (char *) &hostAddress, 0, sizeof(hostAddress) );
  	hostAddress.sin_family = AF_INET;
  	hostAddress.sin_addr.s_addr = inet_addr(host);
  	hostAddress.sin_port = htons(port);
  
  	return hostAddress;	
}



void Channel::closeSocket() {
	::close(ownSocket);
}

int Channel::freePort(int style) {
	int socketID;
	struct sockaddr_in address;
	
// inizializza il  socket
	if (( socketID = socket(AF_INET, style, 0)) < 0) 
    	throwException(Error( "Inizializzazione del socket fallita") );

    char value = 1;
	if (( setsockopt (socketID, SOL_SOCKET, SO_REUSEADDR,  &value,
		    sizeof (int)) ) < 0) 
    	throwException(Error( "Inizializzazione del socket fallita") );


// fa il binding del socket
	memset( (char *) &address, 0, sizeof(address) );
  	address.sin_family = AF_INET;
  	address.sin_addr.s_addr = htonl(INADDR_ANY);
  	address.sin_port = htons(0);
  	if ( bind(socketID, (struct sockaddr *) &address, 
		    sizeof(address)) < 0) 
    	throwException( Error( "Binding of socket failed") );

// Recupera la porta
  	socklen_t socklen = sizeof(address);
  	if ( getsockname(socketID, (struct sockaddr *) &address, 
		    &socklen) < 0) 
    	throwException( Error( "Getting port of socket failed") );
  
	int port = ntohs(address.sin_port);

    ::close(socketID);

    return port;
}

void Channel::readData(void* data, size_t size) {
	if (active == false)			// se non hai apert il pipe
		Error("You must open channel before"); 

	size_t bytesReaden = read(ioSocket, data, size);
	if (bytesReaden < size)
		Error("Read failure "); 
		
}

void Channel::writeData(const void* data, size_t size) {
	if (active == false)			// se non hai apert il pipe
		Error("You must open channel before"); 

	size_t bytesWritten = write(ioSocket, data, size);
	if (bytesWritten < size)
		Error("Write failure "); 
}		

Channel & operator << (Channel & channel, 
		const String & message) {
	size_t len = message.length();
	const char* cstr = message.cString();  
	channel.writeData(&len, sizeof(len));
	channel.writeData(cstr, len+1);
	return channel;
	
}
	
Channel & operator >> (Channel & channel, 
		String & message) {
	size_t len;  
	channel.readData(&len, sizeof(len));
	char *tmpStr=new char[len+8];
	channel.readData(tmpStr, len+1);
	message = tmpStr;
	delete tmpStr;
	return channel;
}

Channel & operator << (Channel & channel, 
		int message) {
	channel.writeData(&message, sizeof(int));
	return channel;
	
}
	
Channel & operator >> (Channel & channel, 
		int & message) {
	channel.readData(&message, sizeof(int));
	return channel;
}

