classdef HH_Disinhibition_GroundUp < handle
    properties
        % UI Components
        Fig, AxV1, AxV2, AxS, AxSweep, StatusLabel, Indicator
        % Sliders
        dt_Slider, g1_Slider, g2N1_Slider, g2N2_Slider, gN1N2_Slider
        dt_Lbl, g1_Lbl, g2N1_Lbl, g2N2_Lbl, gV_Lbl
        % Dropdowns
        Type1_Drop, Type2_Drop, Type3_Drop, Type4_Drop
        % Model Parameters
        C = 1.0; 
        gbar_Na1 = 120; % N1 Na conductance
        gbar_Na2 = 125; % N2 Na conductance 
        gbar_K = 36; gbar_L = 0.3;
        E_Na = 45; E_K = -82; E_L = -59;
        theta = 20; ksyn = 2; 
        alpha_fast = 3; alpha_slow = 1;
        beta_inh = 0.1; beta_exc = 1.0;
        
        Options % Solver settings
    end
    
    methods
        function obj = HH_Disinhibition_GroundUp()
            obj.Options = odeset('RelTol', 1e-6, 'AbsTol', 1e-8, 'MaxStep', 0.05);
            obj.Fig = uifigure('Name', 'HH Flexible Circuit', 'Position', [100 50 1100 900]);
            
            % Axes Setup
            obj.AxV1 = uiaxes(obj.Fig, 'Position', [50 650 500 210]);
            title(obj.AxV1, 'N1: Interneuron Potential'); ylabel(obj.AxV1, 'mV'); grid(obj.AxV1, 'on');
            obj.AxV2 = uiaxes(obj.Fig, 'Position', [50 400 500 210]);
            title(obj.AxV2, 'N2: Output Neuron'); ylabel(obj.AxV2, 'mV'); grid(obj.AxV2, 'on');
            obj.AxS = uiaxes(obj.Fig, 'Position', [50 200 500 180]);
            title(obj.AxS, 'Synaptic Gating'); grid(obj.AxS, 'on');
            obj.AxSweep = uiaxes(obj.Fig, 'Position', [50 40 500 130]);
            title(obj.AxSweep, 'Sweep: max(V2) vs dt'); xlabel(obj.AxSweep, 'dt (ms)'); grid(obj.AxSweep, 'on');

            % Controls
            uilabel(obj.Fig, 'Position', [600 830 200 20], 'Text', 'TIMING OFFSET (dt)', 'FontWeight', 'bold');
            obj.dt_Slider = uislider(obj.Fig, 'Position', [600 810 200 3], 'Limits', [-15 15], 'Value', 0, 'ValueChangedFcn', @(~,~) obj.updatePlot());
            obj.dt_Lbl = uilabel(obj.Fig, 'Position', [820 800 50 20], 'Text', '0.0');

            uilabel(obj.Fig, 'Position', [600 750 250 20], 'Text', 'SYNAPTIC WEIGHTS (0-1) & TYPES', 'FontWeight', 'bold');
            opts = {'Inhibitory (-70mV)', 'Excitatory (+50mV)'};
            vals = [-70, 50];

            % All sliders now have Limits [0 1]
            % Synapse 1: Vpre1 -> N1
            uilabel(obj.Fig, 'Position', [600 720 200 20], 'Text', 'g1: Vpre1 -> N1');
            obj.g1_Slider = uislider(obj.Fig, 'Position', [600 705 180 3], 'Limits', [0 1], 'Value', 0.5, 'ValueChangedFcn', @(~,~) obj.updatePlot());
            obj.Type1_Drop = uidropdown(obj.Fig, 'Position', [800 715 150 22], 'Items', opts, 'ItemsData', vals, 'Value', 50, 'ValueChangedFcn', @(~,~) obj.updatePlot());

            % Synapse 2: Vpre2 -> N1
            uilabel(obj.Fig, 'Position', [600 650 200 20], 'Text', 'g2: Vpre2 -> N1');
            obj.g2N1_Slider = uislider(obj.Fig, 'Position', [600 635 180 3], 'Limits', [0 1], 'Value', 0.3, 'ValueChangedFcn', @(~,~) obj.updatePlot());
            obj.Type2_Drop = uidropdown(obj.Fig, 'Position', [800 645 150 22], 'Items', opts, 'ItemsData', vals, 'Value', 50, 'ValueChangedFcn', @(~,~) obj.updatePlot());

            % Synapse 3: Vpre2 -> N2
            uilabel(obj.Fig, 'Position', [600 580 200 20], 'Text', 'g3: Vpre2 -> N2 (Slow)');
            obj.g2N2_Slider = uislider(obj.Fig, 'Position', [600 565 180 3], 'Limits', [0 1], 'Value', 0.3, 'ValueChangedFcn', @(~,~) obj.updatePlot());
            obj.Type3_Drop = uidropdown(obj.Fig, 'Position', [800 575 150 22], 'Items', opts, 'ItemsData', vals, 'Value', 50, 'ValueChangedFcn', @(~,~) obj.updatePlot());

            % Synapse 4: N1 -> N2
            uilabel(obj.Fig, 'Position', [600 510 200 20], 'Text', 'g4: N1 -> N2');
            obj.gN1N2_Slider = uislider(obj.Fig, 'Position', [600 495 180 3], 'Limits', [0 1], 'Value', 0.5, 'ValueChangedFcn', @(~,~) obj.updatePlot());
            obj.Type4_Drop = uidropdown(obj.Fig, 'Position', [800 505 150 22], 'Items', opts, 'ItemsData', vals, 'Value', 50, 'ValueChangedFcn', @(~,~) obj.updatePlot());

            uibutton(obj.Fig, 'Position', [600 350 200 40], 'Text', 'RUN DT SWEEP', 'BackgroundColor', [0.2 0.6 1], 'FontWeight', 'bold', 'ButtonPushedFcn', @(~,~) obj.runSweep());
            obj.Indicator = uipanel(obj.Fig, 'Position', [600 300 20 20], 'BackgroundColor', 'red');
            obj.StatusLabel = uilabel(obj.Fig, 'Position', [630 300 300 20], 'Text', 'N2 Quiescent');

            obj.updatePlot();
        end

        function updatePlot(obj)
            obj.dt_Lbl.Text = sprintf('%.1f', obj.dt_Slider.Value);
            Y0 = [-69.898, 0.053, 0.592, 0.319, 0, 0, -69.898, 0.053, 0.592, 0.319, 0, 0];
            tspan = 0:0.05:100;
            [T, Y] = ode45(@(t, y) obj.dydt_circuit(t, y, obj.dt_Slider.Value), tspan, Y0, obj.Options);
            plot(obj.AxV1, T, Y(:,1), 'b'); obj.AxV1.YLim = [-85 60];
            plot(obj.AxV2, T, Y(:,7), 'r'); obj.AxV2.YLim = [-85 60];
            cla(obj.AxS); hold(obj.AxS, 'on');
            plot(obj.AxS, T, Y(:,5), 'b', 'DisplayName', 's1'); plot(obj.AxS, T, Y(:,6), 'Color', [0.9 0.6 0], 'DisplayName', 's2');
            plot(obj.AxS, T, Y(:,11), 'm--', 'DisplayName', 's3 (Slow)'); plot(obj.AxS, T, Y(:,12), 'g', 'LineWidth', 1.2, 'DisplayName', 'sV');
            legend(obj.AxS, 'Location', 'northeast', 'FontSize', 8); hold(obj.AxS, 'off');
            if any(Y(T > 40, 7) > 0), obj.Indicator.BackgroundColor = [0 1 0]; obj.StatusLabel.Text = 'N2 Spike Detected!';
            else, obj.Indicator.BackgroundColor = [1 0 0]; obj.StatusLabel.Text = 'N2 Sub-threshold'; end
        end

        function runSweep(obj)
            dt_vec = -10:0.5:10; vmax = zeros(size(dt_vec));
            Y0 = [-69.898, 0.053, 0.592, 0.319, 0, 0, -69.898, 0.053, 0.592, 0.319, 0, 0];
            tspan = 0:0.05:100;
            for i = 1:length(dt_vec)
                [T, Y] = ode45(@(t, y) obj.dydt_circuit(t, y, dt_vec(i)), tspan, Y0, obj.Options);
                vmax(i) = max(Y(T > 40, 7));
            end
            cla(obj.AxSweep); hold(obj.AxSweep, 'on'); isAP = vmax > 0;
            plot(obj.AxSweep, dt_vec(~isAP), vmax(~isAP), 'ro', 'MarkerFaceColor', 'r');
            plot(obj.AxSweep, dt_vec(isAP), vmax(isAP), 'go', 'MarkerFaceColor', 'g');
            yline(obj.AxSweep, 0, 'k--'); hold(obj.AxSweep, 'off');
        end

        function dY = dydt_circuit(obj, t, Y, dt_val)
            V1=Y(1); m1=Y(2); h1=Y(3); n1=Y(4); s1=Y(5); s2=Y(6);
            V2=Y(7); m2=Y(8); h2=Y(9); n2=Y(10); s3=Y(11); sV=Y(12);

            Vp1 = -70 + 90 * (t >= 50 && t <= 52);
            Vp2 = -70 + 90 * (t >= (50 + dt_val) && t <= (50 + dt_val + 2));

            [m1i, tm1] = obj.hh_vars(V1, 'm'); [h1i, th1] = obj.hh_vars(V1, 'h'); [n1i, tn1] = obj.hh_vars(V1, 'n');
            [m2i, tm2] = obj.hh_vars(V2, 'm'); [h2i, th2] = obj.hh_vars(V2, 'h'); [n2i, tn2] = obj.hh_vars(V2, 'n');

            % Beta values adjust dynamically based on reversal potential
            b1 = ifthen(obj.Type1_Drop.Value == 50, obj.beta_exc, obj.beta_inh);
            b2 = ifthen(obj.Type2_Drop.Value == 50, obj.beta_exc, obj.beta_inh);
            b3 = ifthen(obj.Type3_Drop.Value == 50, obj.beta_exc, obj.beta_inh);
            b4 = ifthen(obj.Type4_Drop.Value == 50, obj.beta_exc, obj.beta_inh);

            dY = zeros(12,1);
            dY(1) = (-1/obj.C) * (obj.gbar_Na1*m1^3*h1*(V1-obj.E_Na) + obj.gbar_K*n1^4*(V1-obj.E_K) + ...
                    obj.gbar_L*(V1-obj.E_L) + obj.g1_Slider.Value*s1*(V1-obj.Type1_Drop.Value) + obj.g2N1_Slider.Value*s2*(V1-obj.Type2_Drop.Value));
            dY(2:4) = [(m1i-m1)/tm1; (h1i-h1)/th1; (n1i-n1)/tn1];
            
            dY(7) = (-1/obj.C) * (obj.gbar_Na2*m2^3*h2*(V2-obj.E_Na) + obj.gbar_K*n2^4*(V2-obj.E_K) + ...
                    obj.gbar_L*(V2-obj.E_L) + obj.g2N2_Slider.Value*s3*(V2-obj.Type3_Drop.Value) + obj.gN1N2_Slider.Value*sV*(V2-obj.Type4_Drop.Value));
            dY(8:10) = [(m2i-m2)/tm2; (h2i-h2)/th2; (n2i-n2)/tn2];

            dY(5) = obj.alpha_fast / (1 + exp((obj.theta-Vp1)/obj.ksyn)) * (1-s1) - b1 * s1;
            dY(6) = obj.alpha_fast / (1 + exp((obj.theta-Vp2)/obj.ksyn)) * (1-s2) - b2 * s2;
            dY(11) = obj.alpha_slow / (1 + exp((obj.theta-Vp2)/obj.ksyn)) * (1-s3) - b3 * s3;
            dY(12) = obj.alpha_fast / (1 + exp((obj.theta-V1)/obj.ksyn)) * (1-sV) - b4 * sV;
        end

        function [inf, tau] = hh_vars(~, V, type)
            if type=='m', a=.1*(V+45)/(1-exp(-(V+45)/10)); b=4*exp(-(V+70)/18);
            elseif type=='h', a=.07*exp(-(V+70)/20); b=1/(1+exp(-(V+40)/10));
            else, a=.01*(V+60)/(1-exp(-(V+60)/10)); b=.125*exp(-(V+70)/80); end
            inf=a/(a+b); tau=1/(a+b);
        end
    end
end

function val = ifthen(cond, a, b)
    if cond, val = a; else, val = b; end
end