classdef HH_DualSynapse_Sweep_GUI < handle
    properties
        % UI Components
        Fig
        AxV, AxS, AxSweep
        StatusLabel, Indicator
        % Sliders and Controls
        I0_Slider, dt_Slider
        gsyn1_Slider, gsyn2_Slider
        betas1_Slider, betas2_Slider
        Vsyn1_Drop, Vsyn2_Drop
        % Value Labels
        I0_Lbl, dt_Lbl, g1_Lbl, g2_Lbl, b1_Lbl, b2_Lbl
        % Model Parameters
        C = 1.0; gbar_Na = 120; gbar_K = 36; gbar_L = 0.3;
        E_Na = 45; E_K = -82; E_L = -59;
        theta = 20; ksyn = 2; alphas = 3; dur = 2;        
        Options % Solver settings
    end
    
    methods
        function obj = HH_DualSynapse_Sweep_GUI()
            % Robust solver options
            obj.Options = odeset('RelTol', 1e-6, 'AbsTol', 1e-8, 'MaxStep', 0.05);

            % Create Figure
            obj.Fig = uifigure('Name', 'HH Dual Synapse: Summation Sweep', 'Position', [100 50 1000 900]);
            
            % Plotting Areas
            obj.AxV = uiaxes(obj.Fig, 'Position', [50 630 530 220]);
            title(obj.AxV, 'Membrane Potential (V)'); ylabel(obj.AxV, 'mV'); grid(obj.AxV, 'on');
            
            obj.AxS = uiaxes(obj.Fig, 'Position', [50 360 530 220]);
            title(obj.AxS, 'Synaptic Gating (s1: Blue, s2: Orange)'); ylabel(obj.AxS, 's'); grid(obj.AxS, 'on');

            obj.AxSweep = uiaxes(obj.Fig, 'Position', [50 70 530 220]);
            title(obj.AxSweep, 'Maximum Voltage vs. Timing Offset (dt)');
            xlabel(obj.AxSweep, 'dt (ms)'); ylabel(obj.AxSweep, 'max(V) (mV)'); grid(obj.AxSweep, 'on');

            % --- Analysis & Sweep Controls ---
            uilabel(obj.Fig, 'Position', [650 140 150 20], 'Text', 'Summation Result:', 'FontWeight', 'bold');
            obj.Indicator = uipanel(obj.Fig, 'Position', [770 140 20 20], 'BackgroundColor', 'red');
            obj.StatusLabel = uilabel(obj.Fig, 'Position', [800 140 200 20], 'Text', 'Sub-threshold');

            uibutton(obj.Fig, 'Position', [650 80 200 40], 'Text', 'RUN DT SWEEP', ...
                'BackgroundColor', [0.2 0.6 1.0], 'FontWeight', 'bold', 'FontColor', 'white', ...
                'ButtonPushedFcn', @(~,~) obj.runSweep());

            % --- Control Panel (Right Side) ---
            % Global
            uilabel(obj.Fig, 'Position', [650 830 150 20], 'Text', 'External Current (I0)', 'FontWeight', 'bold');
            obj.I0_Slider = uislider(obj.Fig, 'Position', [650 815 200 3], 'Limits', [-10 20], 'Value', 0, 'ValueChangedFcn', @(~,~) obj.updatePlot());
            obj.I0_Lbl = uilabel(obj.Fig, 'Position', [870 805 50 20], 'Text', '0.0');

            uilabel(obj.Fig, 'Position', [650 760 150 20], 'Text', 'Timing Offset (dt)', 'FontWeight', 'bold');
            obj.dt_Slider = uislider(obj.Fig, 'Position', [650 745 200 3], 'Limits', [-10 10], 'Value', 0, 'ValueChangedFcn', @(~,~) obj.updatePlot());
            obj.dt_Lbl = uilabel(obj.Fig, 'Position', [870 735 50 20], 'Text', '0.0');

            % Neuron 1
            uilabel(obj.Fig, 'Position', [650 680 200 20], 'Text', 'NEURON 1', 'FontColor', [0 0.45 0.74], 'FontWeight', 'bold');
            uilabel(obj.Fig, 'Position', [650 660 200 20], 'Text', 'gsyn1');
            obj.gsyn1_Slider = uislider(obj.Fig, 'Position', [650 645 200 3], 'Limits', [0 2], 'Value', 0.03, 'ValueChangedFcn', @(~,~) obj.updatePlot());
            obj.g1_Lbl = uilabel(obj.Fig, 'Position', [870 635 50 20], 'Text', '0.03');
            uilabel(obj.Fig, 'Position', [650 590 200 20], 'Text', 'betas1');
            obj.betas1_Slider = uislider(obj.Fig, 'Position', [650 575 200 3], 'Limits', [0.01 1], 'Value', 1.0, 'ValueChangedFcn', @(~,~) obj.updatePlot());
            obj.b1_Lbl = uilabel(obj.Fig, 'Position', [870 585 50 20], 'Text', '1.00');
            obj.Vsyn1_Drop = uidropdown(obj.Fig, 'Position', [650 515 200 25], 'Items', {'Excitatory (+70)', 'Inhibitory (-70)'}, 'ItemsData', [70, -70], 'ValueChangedFcn', @(~,~) obj.updatePlot());

            % Neuron 2
            uilabel(obj.Fig, 'Position', [650 460 200 20], 'Text', 'NEURON 2', 'FontColor', [0.85 0.33 0.1], 'FontWeight', 'bold');
            uilabel(obj.Fig, 'Position', [650 440 200 20], 'Text', 'gsyn2');
            obj.gsyn2_Slider = uislider(obj.Fig, 'Position', [650 425 200 3], 'Limits', [0 2], 'Value', 0.03, 'ValueChangedFcn', @(~,~) obj.updatePlot());
            obj.g2_Lbl = uilabel(obj.Fig, 'Position', [870 415 50 20], 'Text', '0.03');
            uilabel(obj.Fig, 'Position', [650 370 200 20], 'Text', 'betas2');
            obj.betas2_Slider = uislider(obj.Fig, 'Position', [650 355 200 3], 'Limits', [0.01 1], 'Value', 1.0, 'ValueChangedFcn', @(~,~) obj.updatePlot());
            obj.b2_Lbl = uilabel(obj.Fig, 'Position', [870 365 50 20], 'Text', '1.00');
            obj.Vsyn2_Drop = uidropdown(obj.Fig, 'Position', [650 295 200 25], 'Items', {'Excitatory (+70)', 'Inhibitory (-70)'}, 'ItemsData', [70, -70], 'ValueChangedFcn', @(~,~) obj.updatePlot());

            obj.updatePlot();
        end

        function runSweep(obj)
            dt_vec = -10:0.5:10;
            maxV = zeros(size(dt_vec));
            isAP = false(size(dt_vec));
            
            % Use current GUI settings for the sweep
            Y0 = [-69.898, 0.053, 0.592, 0.319, 0, 0];
            tspan = 0:0.05:100;
            
            for i = 1:length(dt_vec)
                temp_dt = dt_vec(i);
                [T, Y] = ode45(@(t, y) obj.dydt_sweep(t, y, temp_dt), tspan, Y0, obj.Options);
                maxV(i) = max(Y(:,1));
                isAP(i) = maxV(i) > 0;
            end
            
            % Plot Sweep Results
            cla(obj.AxSweep);
            hold(obj.AxSweep, 'on');
            % Plot sub-threshold as blue dots, APs as red stars
            plot(obj.AxSweep, dt_vec(~isAP), maxV(~isAP), 'bo', 'MarkerFaceColor', 'b', 'DisplayName', 'No Spike');
            plot(obj.AxSweep, dt_vec(isAP), maxV(isAP), 'r*', 'MarkerSize', 8, 'DisplayName', 'AP Triggered');
            yline(obj.AxSweep, 0, '--', 'Threshold (0mV)');
            legend(obj.AxSweep, 'Location', 'best');
            hold(obj.AxSweep, 'off');
        end
        
        function updatePlot(obj)
            obj.I0_Lbl.Text = sprintf('%.1f', obj.I0_Slider.Value);
            obj.dt_Lbl.Text = sprintf('%.1f', obj.dt_Slider.Value);
            obj.g1_Lbl.Text = sprintf('%.2f', obj.gsyn1_Slider.Value);
            obj.g2_Lbl.Text = sprintf('%.2f', obj.gsyn2_Slider.Value);
            obj.b1_Lbl.Text = sprintf('%.2f', obj.betas1_Slider.Value);
            obj.b2_Lbl.Text = sprintf('%.2f', obj.betas2_Slider.Value);

            Y0 = [-69.898, 0.053, 0.592, 0.319, 0, 0];
            tspan = 0:0.05:100;
            [T, Y] = ode45(@(t, y) obj.dydt_dual(t, y), tspan, Y0, obj.Options);
            
            plot(obj.AxV, T, Y(:,1), 'k', 'LineWidth', 1.2); obj.AxV.YLim = [-85 60];
            plot(obj.AxS, T, Y(:,5), 'Color', [0 0.45 0.74], 'LineWidth', 1.5); hold(obj.AxS, 'on');
            plot(obj.AxS, T, Y(:,6), 'Color', [0.85 0.33 0.1], 'LineWidth', 1.5); hold(obj.AxS, 'off');
            obj.AxS.YLim = [0 1.1];

            if any(Y(T > 45, 1) > 0)
                obj.Indicator.BackgroundColor = [0 1 0]; obj.StatusLabel.Text = 'Spike Detected';
            else
                obj.Indicator.BackgroundColor = [1 0 0]; obj.StatusLabel.Text = 'Sub-threshold';
            end
        end
        
        function dY = dydt_dual(obj, t, Y)
            dY = obj.calc_ode(t, Y, obj.dt_Slider.Value);
        end

        function dY = dydt_sweep(obj, t, Y, dt_val)
            dY = obj.calc_ode(t, Y, dt_val);
        end

        function dY = calc_ode(obj, t, Y, dt_val)
            V = Y(1); m = Y(2); h = Y(3); n = Y(4); s1 = Y(5); s2 = Y(6);
            V_pre1 = 90 * (t >= 50 && t <= 52) - 70;
            V_pre2 = 90 * (t >= (50 + dt_val) && t <= (50 + dt_val + 2)) - 70;
            [m_inf, tau_m] = obj.m_and_tau_m(V); [h_inf, tau_h] = obj.h_and_tau_h(V); [n_inf, tau_n] = obj.n_and_tau_n(V);
            dY = zeros(6,1);
            dY(1) = (-1/obj.C) * (obj.gbar_Na*m^3*h*(V-obj.E_Na) + obj.gbar_K*n^4*(V-obj.E_K) + ...
                                  obj.gbar_L*(V-obj.E_L) - obj.I0_Slider.Value + ...
                                  obj.gsyn1_Slider.Value * s1 * (V - obj.Vsyn1_Drop.Value) + ...
                                  obj.gsyn2_Slider.Value * s2 * (V - obj.Vsyn2_Drop.Value));
            dY(2) = (1/tau_m) * (m_inf - m); dY(3) = (1/tau_h) * (h_inf - h); dY(4) = (1/tau_n) * (n_inf - n);
            dY(3) = (1/tau_h) * (h_inf - h);
            dY(4) = (1/tau_n) * (n_inf - n);
            dY(5) = obj.alphas / (1 + exp((obj.theta - V_pre1)/obj.ksyn)) * (1 - s1) - obj.betas1_Slider.Value * s1;
            dY(6) = obj.alphas / (1 + exp((obj.theta - V_pre2)/obj.ksyn)) * (1 - s2) - obj.betas2_Slider.Value * s2;
        end

        function [m_inf, tau_m] = m_and_tau_m(~, V)
            a = ((V+45)/10) / (1-exp(-(V+45)/10)); b = 4*exp(-(V+70)/18);
            m_inf = a/(a+b); tau_m = 1/(a+b);
        end
        function [h_inf, tau_h] = h_and_tau_h(~, V)
            a = 0.07*exp(-(V+70)/20); b = 1/(1+exp(-(V+40)/10));
            h_inf = a/(a+b); tau_h = 1/(a+b);
        end
        function [n_inf, tau_n] = n_and_tau_n(~, V)
            a = 0.1*((V+60)/10) / (1-exp(-(V+60)/10)); b = 0.125*exp(-(V+70)/80);
            n_inf = a/(a+b); tau_n = 1/(a+b);
        end
    end
end