classdef HH_Synapse_GUI < handle
    properties
        % UI Components
        Fig
        AxV
        AxS
        % Sliders and Controls
        I0_Slider, gsyn_Slider, betas_Slider, dur_Slider
        Vsyn_Dropdown
        % Model Parameters (Defaults)
        C = 1.0; gbar_Na = 120; gbar_K = 36; gbar_L = 0.3;
        E_Na = 45; E_K = -82; E_L = -59;
        theta = 20; ksyn = 2; alphas = 3;
        % User-adjustable defaults
        I0 = 10; gsyn = 2; Vsyn = 70; betas = 0.05; dur = 2;
        Options % Solver settings
    end
    
    methods
        function obj = HH_Synapse_GUI()
            % Robust solver options
            obj.Options = odeset('RelTol', 1e-6, 'AbsTol', 1e-8, 'MaxStep', 0.05);

            % Create Figure
            obj.Fig = uifigure('Name', 'Hodgkin-Huxley Synaptic Model', 'Position', [100 100 800 600]);
            
            % Create Axes for Voltage
            obj.AxV = uiaxes(obj.Fig, 'Position', [50 350 500 200]);
            title(obj.AxV, 'Membrane Potential (V)');
            ylabel(obj.AxV, 'mV');
            
            % Create Axes for Synaptic Variable (s)
            obj.AxS = uiaxes(obj.Fig, 'Position', [50 100 500 200]);
            title(obj.AxS, 'Synaptic Gating Variable (s)');
            ylabel(obj.AxS, 's');
            xlabel(obj.AxS, 'Time (ms)');

            % --- Control Panel ---
            uilabel(obj.Fig, 'Position', [600 530 150 20], 'Text', 'External Current (I0)', 'FontWeight', 'bold');
            obj.I0_Slider = uislider(obj.Fig, 'Position', [600 510 150 3], 'Limits', [-10 20], 'Value', 10, ...
                'ValueChangedFcn', @(src, event) obj.updatePlot());

            uilabel(obj.Fig, 'Position', [600 450 150 20], 'Text', 'Synaptic Conduct. (gsyn)', 'FontWeight', 'bold');
            obj.gsyn_Slider = uislider(obj.Fig, 'Position', [600 430 150 3], 'Limits', [0 2], 'Value', 2, ...
                'ValueChangedFcn', @(src, event) obj.updatePlot());

            uilabel(obj.Fig, 'Position', [600 370 150 20], 'Text', 'Decay Rate (betas)', 'FontWeight', 'bold');
            obj.betas_Slider = uislider(obj.Fig, 'Position', [600 350 150 3], 'Limits', [0.01 1], 'Value', 0.05, ...
                'ValueChangedFcn', @(src, event) obj.updatePlot());

            uilabel(obj.Fig, 'Position', [600 290 150 20], 'Text', 'Pulse Duration (dur)', 'FontWeight', 'bold');
            obj.dur_Slider = uislider(obj.Fig, 'Position', [600 270 150 3], 'Limits', [0 10], 'Value', 2, ...
                'ValueChangedFcn', @(src, event) obj.updatePlot());

            uilabel(obj.Fig, 'Position', [600 210 150 20], 'Text', 'Synaptic Type (Vsyn)', 'FontWeight', 'bold');
            obj.Vsyn_Dropdown = uidropdown(obj.Fig, 'Position', [600 180 150 25], ...
                'Items', {'Excitatory (+70)', 'Inhibitory (-70)'}, 'ItemsData', [70, -70], ...
                'ValueChangedFcn', @(src, event) obj.updatePlot());

            % Initial Plot
            obj.updatePlot();
        end
        
        function updatePlot(obj)
            % Get values from UI
            obj.I0 = obj.I0_Slider.Value;
            obj.gsyn = obj.gsyn_Slider.Value;
            obj.betas = obj.betas_Slider.Value;
            obj.dur = obj.dur_Slider.Value;
            obj.Vsyn = obj.Vsyn_Dropdown.Value;

            % Run Simulation
            Y0 = [-69.8982, 0.0535, 0.5925, 0.3192, 0];
            tspan = 0:0.01:200;
            
            [T, Y] = ode45(@(t, y) obj.dydt_HH(t, y), tspan, Y0, obj.Options);
            
            % Update Plots
            plot(obj.AxV, T, Y(:,1), 'b', 'LineWidth', 1.5);
            obj.AxV.YLim = [-85 50];
            grid(obj.AxV, 'on');
            
            plot(obj.AxS, T, Y(:,5), 'r', 'LineWidth', 1.5);
            obj.AxS.YLim = [0 1];
            grid(obj.AxS, 'on');
        end
        
        function dY = dydt_HH(obj, t, Y)
            V1 = Y(1); m1 = Y(2); h1 = Y(3); n1 = Y(4); s1 = Y(5);
            
            % Presynaptic input (V2)
            V2 = 90 * (t >= 50 && t <= 50 + obj.dur) - 70;

            [m_inf1, tau_m1] = obj.m_and_tau_m(V1);
            [h_inf1, tau_h1] = obj.h_and_tau_h(V1);
            [n_inf1, tau_n1] = obj.n_and_tau_n(V1);

            dY = zeros(5,1);
            % Membrane Equation
            dY(1) = (-1/obj.C) * (obj.gbar_Na*m1^3*h1*(V1-obj.E_Na) + ...
                                  obj.gbar_K*n1^4*(V1-obj.E_K) + ...
                                  obj.gbar_L*(V1-obj.E_L) - obj.I0 + ...
                                  obj.gsyn*s1*(V1-obj.Vsyn));
            % Gating Variables
            dY(2) = (1/tau_m1) * (m_inf1 - m1);
            dY(3) = (1/tau_h1) * (h_inf1 - h1);
            dY(4) = (1/tau_n1) * (n_inf1 - n1);
            % Synaptic Variable
            dY(5) = obj.alphas / (1 + exp((obj.theta - V2)/obj.ksyn)) * (1 - s1) - obj.betas * s1;
        end
        
        % --- Helper Functions (HH Math) ---
        function [m_inf, tau_m] = m_and_tau_m(~, V)
            alpha_m = ((V+45)/10) / (1-exp(-(V+45)/10));
            beta_m = 4*exp(-(V+70)/18);
            m_inf = alpha_m / (alpha_m + beta_m);
            tau_m = 1 / (alpha_m + beta_m);
        end
        
        function [h_inf, tau_h] = h_and_tau_h(~, V)
            alpha_h = 0.07*exp(-(V+70)/20);
            beta_h = 1 / (1 + exp(-(V+40)/10));
            h_inf = alpha_h / (alpha_h + beta_h);
            tau_h = 1 / (alpha_h + beta_h);
        end
        
        function [n_inf, tau_n] = n_and_tau_n(~, V)
            alpha_n = 0.1*((V+60)/10) / (1-exp(-(V+60)/10));
            beta_n = 0.125*exp(-(V+70)/80);
            n_inf = alpha_n / (alpha_n + beta_n);
            tau_n = 1 / (alpha_n + beta_n);
        end
    end
end