/*
 * Copyright (c) 2008 Department of Information Engineering, University of Padova, Italy
 * Contributors: Giovanni Zanca, Nicola Bui, Riccardo Crepaldi and Michele Rossi
 * All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 *       copyright notice, this list of conditions and the following
 *       disclaimer in the documentation and/or other materials provided
 *       with the distribution.
 *     * The name of the author may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
 

module DumpToLedsP
{
    provides {
        interface DumpToLeds;
        interface Init;
    }
    uses {
        interface GeneralIO as Led0;
        interface GeneralIO as Led1;
        interface GeneralIO as Led2;
        //interface GeneralIO as Led3;
    }
}

#if PLATFORM_TELOSB
	#define set0 Led0.clr
	#define set1 Led1.clr
	#define set2 Led2.clr
	#define clr0 Led0.set
	#define clr1 Led1.set
	#define clr2 Led2.set
#else
	#define set0 Led0.set
	#define set1 Led1.set
	#define set2 Led2.set
	#define clr0 Led0.clr
	#define clr1 Led1.clr
	#define clr2 Led2.clr
#endif

implementation
{
	uint8_t m_status=0;
	void getLedsStatus();
	void restoreLedsStatus();

    command error_t Init.init()
    {
        atomic {
            call Led0.makeOutput();
            call Led1.makeOutput();
            call Led2.makeOutput();
            call clr0();
            call clr1();
            call clr2();
        }
        return SUCCESS;
    }

    void _wait_( uint8_t time )
    {
        uint16_t i;
        uint16_t k;
        for(i=time; i>0; i--) {
            for(k=0xC350; k>0; k--) {__asm__ __volatile__("; loop %0": "+r" (k));}
//            {
//                __asm volatile ("nop\n\t"
//                     "nop\n\t"
//                     "nop\n\t"
//                    "nop\n\t");
//            }
        }
    }

	void getLedsStatus(){
		if(!(call Led0.get())) m_status|=0x1;
		if(!(call Led1.get())) m_status|=(0x2);
		if(!(call Led2.get())) m_status|=(0x4);
	}

	void restoreLedsStatus(){
		if(m_status & 0x1) call set0();
		else call clr0();
		if(m_status & (0x2)) call set1();
		else call clr1();
		if(m_status & (0x4)) call set2();
		else call clr2();
	}


    command void DumpToLeds.dump4bit( uint8_t n, uint8_t time )
    {
        if(n&0x1) {
            call set0();
        } else {
            call clr0();
        }
        if(n&0x2) {
            call set1();
        } else {
            call clr1();
        }
        if(n&0x4) {
            call set2();
        } else {
            call clr2();
        }
        if(n&0x8) {
            //call Led3.set();
        } else {
            //call Led3.clr();
        }
        if(time>0) {
            _wait_( time );
        }
    }

    command void DumpToLeds.dump4bitR( uint8_t n, uint8_t time )
    {
		getLedsStatus();
        call DumpToLeds.dump4bit( n, time );
        call DumpToLeds.dump4bit( 0x0, time );
		restoreLedsStatus();
    }

    command void DumpToLeds.dump8bit( uint8_t n, uint8_t time )
    {
        call DumpToLeds.dump4bit( n, time );
        call DumpToLeds.dump4bit( n>>4, time );
    }

    command void DumpToLeds.dump8bitR( uint8_t n, uint8_t time )
    {
		getLedsStatus();
        call DumpToLeds.dump4bitR( n, time );
        call DumpToLeds.dump4bitR( n>>4, time );
		restoreLedsStatus();
    }

    command void DumpToLeds.dump16bit( uint16_t n, uint8_t time )
    {
        call DumpToLeds.dump8bit( (uint8_t)n, time );
        call DumpToLeds.dump8bit( (uint8_t)(n>>8), time );
    }

    command void DumpToLeds.dump16bitR( uint16_t n, uint8_t time )
    {
		getLedsStatus();
        call DumpToLeds.dump8bitR( (uint8_t)n, time );
        call DumpToLeds.dump8bitR( (uint8_t)n>>8, time );
		restoreLedsStatus();
    }

    command void DumpToLeds.dump32bitR( uint32_t n, uint8_t time )
    {
		getLedsStatus();
        call DumpToLeds.dump16bitR( (uint16_t)n, time );
        call DumpToLeds.dump16bitR( (uint16_t)n>>16, time );
		restoreLedsStatus();
    }
}
