/*
 * Copyright (c) 2008 Department of Information Engineering, University of Padova, Italy
 * Contributors: Giovanni Zanca, Nicola Bui, Riccardo Crepaldi and Michele Rossi
 * All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 *       copyright notice, this list of conditions and the following
 *       disclaimer in the documentation and/or other materials provided
 *       with the distribution.
 *     * The name of the author may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "bootloader.h"
#include "Synapse.h"
#include "external_flash.h"
#include "Synapse_msgs.h"
#include "Synapse_Partition.h"
//#include "printf.h"
#include <Timer.h>

#define RAND_SLOT (call Random.rand16()%(SLOT_NUM/2))
#define INVALID_APP_ID 0
#define INVALID_BLOCK_ID 0
#define BLOCK_PAYLOAD_SIZE (SYMBOLS_PER_BLOCK*SNP_DATA_SIZE-CRC_SIZE)


module SyNaPseP
{
	provides interface Init;
	uses {
		interface Boot;
		interface Random;
		interface SplitControl as AMControl;
		interface SplitControl as SerialControl;
		interface Packet as SerialPacket;
		interface Receive as SerialReceive;
		
		interface TimeSyncAMSend<TMilli, uint32_t> as AdvSend;
		interface TimeSyncPacket<TMilli, uint32_t> as TimeSyncAdvPacket;
		interface Packet as AdvPacket;
		interface AMPacket as AdvAMPacket;
		interface Receive as AdvReceive;
		interface Packet as CmdPacket;
		interface AMPacket as CmdAMPacket;
		interface AMSend as CmdSend;
		interface Receive as CmdReceive;
		
		interface Packet as PingPacket;
		interface AMPacket as PingAMPacket;
		interface AMSend as PingSend;
		interface Receive as PingReceive;
		interface Receive as SerialPingReceive;
		
		interface AMSend as ReqSend;
		interface Packet as ReqPacket;
		interface AMPacket as ReqAMPacket;
		interface Receive as ReqReceive;

		interface Leds;
		interface Synapse_BootloaderCommunication as BootloaderCommunication;

		interface Timer<TMilli> as Timer;
		interface Timer<TMilli> as TxTimer;
		#ifdef STAT
		interface Timer<TMilli> as StatTimer;
		#endif
		interface Synapse_Radio;
		interface Crc;

		interface Synapse_PartitionStorage;
		interface BlockRead;
		interface BlockWrite;
		
		command uint16_t get_NumOfImg();
#ifdef LOADABLE
#warning LOADABLE flag is turned on
		interface CC2420Config;
		async command void setAmAddress(am_addr_t a);
#endif
#ifdef STAT
		interface LogWrite;
#endif
#ifdef DEBUG_SYNAPSE
		interface DumpToLeds;
#endif
		interface Synapse_SerialQueue;

	}
}

implementation
{
	enum {  
		SNP_INIT		=0,
		SNP_IDLE		=1,
		SNP_CMD			=2,
		SNP_ADV			=3,
		SNP_REQ			=4,
		SNP_DATA		=5,
    };

	enum {
		FLASH_IDLE		=0,
		FLASH_MOUNTING	=1,
		FLASH_CREATING	=2,
		FLASH_READING	=3,
		FLASH_WRITING	=4,
	};

#ifdef DEBUG_SYNAPSE
	enum {
		DBG_ADV_SNT = 1,
		DBG_REQ_SNT = 2,
		DBG_DATA_SNT= 3,
		DBG_ADV_RCV = 4,
		DBG_REQ_RCV = 5,
		DBG_DATA_RCV= 6,
		DBG_TIMES	= 7,
		DBG_FLASH_MOUNTING	= 8,
		DBG_FLASH_MOUNTED		= 9,
		DBG_FLASH_READING		= 10,
		DBG_FLASH_READ			= 11,
		DBG_FLASH_BUSY			= 12,
		DBG_FLASH_CREATING		= 13,
		DBG_FLASH_CREATED			= 14,
		DBG_INI_REC = 16,
		DBG_INI_SND = 17,
		DBG_DATA_NAK = 18,
		DBG_FLASH_WRITTEN = 19,
		DBG_FLASH_WRITING = 20,
		DBG_CRC_FAILED	= 21,
		DBG_SFD_DELAY = 22,
		DBG_WRONG_ADV = 23,
		DBG_WRONG_REQ = 24,
		DBG_WRONG_DATA = 25,
	};
#endif

/* Tasks and Procedures*/
	task void executeCmdTask();
	void SendStatMsg(uint8_t type,uint32_t a,uint32_t b,uint16_t c,uint16_t d,uint8_t e,uint8_t f);
	void changeStatus(uint8_t newStatus,error_t error);
	void pingManage(message_t* msg);
	uint8_t findPartition(uint16_t appId);
	void syncOnLastSfdRx(uint32_t sfdRxTime, uint32_t RxEventTime, uint32_t baseTime);
	uint8_t getNextBlock(uint8_t *bitMask);
	uint8_t getLastBlock();
	void setBlock(uint8_t block, bool present);
	bool blockAvailable(uint8_t block);

/*Global variables*/
	uint8_t reqRx=0;
	message_t m_adv_msg;
	message_t m_cmd_msg;
	message_t m_req_msg;
	am_addr_t mySource;
	am_addr_t pingSrc;
	uint16_t m_bufferSize;
	uint8_t *m_buffer;
	uint8_t m_blockInBuffer=INVALID_BLOCK_ID;
	uint16_t myAppId=INVALID_APP_ID;
	uint16_t myAppSize=0;
	uint16_t myAppOffset=0x0000;
	uint8_t  myAppBlocks=0;
	uint8_t myBlocks[MAX_BLOCKS_DIV8];
	uint8_t lastSent=0;
	uint8_t blockToSend=0;
	uint8_t myHopCount=0xFF;
	uint8_t status=SNP_IDLE;
	uint8_t flashStatus=FLASH_IDLE;
	bool baseStation=FALSE;
	bool serialConnected=FALSE;
	bool readyToExecute=FALSE;
	bool sending=FALSE;
	uint8_t m_cmd=invalidCmd;
	static const uint16_t adv_time = (ADV_TX_TIME_uS>>10)*TIMER_FREQ; //~0 ms
	static const uint16_t req_time = (REQ_TX_TIME_uS>>10)*TIMER_FREQ; //~0 ms
	static const uint16_t data_time =(DATA_TX_TIME_uS>>10)*TIMER_FREQ; //~0 ms
	static const uint16_t adv_slot=2*TIMER_FREQ;//adv slot duration in timer ticks
	static const uint16_t req_slot=2*TIMER_FREQ;//req slot duration in timer ticks
// 	static const uint16_t data_slot=1;//DATA_TX_TIME_uS>>6;
	static const uint32_t total_data_transmission_time= TOTAL_DATA_TRANSMISSION_TIME;  //(DATA_TX_TIME_uS*SYMBOLS_TO_SEND)>>10; 
	static const uint32_t decode_time=MAX_DECODE_TIME;

	static const uint16_t seeds[NUM_OF_SEEDS] = 
	{0x038C,0x0B9F,	0x0D80,	0x0130F,0x0E65,	0x134B,
	0x0E07,	0x1355,	0x0485,	0x0A79,	0x0EF8,	0x1290,
	0x0E54,	0x06C9,	0x061C,	0x0E88,	0x09A5,	0x1049,
	0x1088,	0x10B0,	0x10EC,	0x123B,	0x126A,	0x067A,
	0x130B,	0x0688,	0x0692,	0x0A76,	0x06FC,	0x09A6,
	0x09DA,	0x09F8,	0x0A82,	0x0ADE,	0x0B10};

	bool reqFlag = FALSE;
	uint8_t reqTxSlot = 0;
	uint8_t retryFlag = 0;
	bool advFlag = FALSE;
	uint8_t advCycles = 0;
	uint8_t advTxSlot = 0;
	bool dataFlag = FALSE;
	uint8_t reqDelayedFlag = 0;
	uint8_t advDelayedFlag = 0;
	uint8_t cmdSent=0;
	uint16_t lastSfdTxTime=0;
	uint16_t lastSfdRxTime=0;


	uint8_t backoffSize = 1;
	uint8_t backoffCount = 0;
	bool advCheck = FALSE;
	uint8_t dbgBlock = 0;
	bool ended = FALSE;		

	#ifdef STAT
	uint16_t tmp_statistics [7];
	bool stat_active_collection=FALSE;
	bool stat_req_sent=FALSE;
	uint16_t stat_adv_tx=0;
	uint16_t stat_req_tx=0;
	uint16_t stat_data_tx=0;
	uint16_t stat_cycles=0;
	uint16_t stat_fixed_rx=0;
	uint16_t stat_failed_rx=0;
	uint8_t stat_min_hopcount=0xFF;
	uint8_t stat_max_hopcount=0;
	#endif

	event void Boot.booted()
	{
		uint32_t i;
		//call Leds.set(1);
		for(i=0;i<0xFFFFF;i++);
		call AMControl.start();
	}

	event void AMControl.startDone(error_t err)
	{
		if(err!=SUCCESS) {
			call AMControl.start();
		}
		else{
			call SerialControl.start();
		}
	}

	event void SerialControl.startDone(error_t err)
	{
		if(err!=SUCCESS) {
			call SerialControl.start();
		}
		else{
			changeStatus(SNP_IDLE,SUCCESS);
			m_buffer = call Synapse_Radio.getBuffer( &m_bufferSize );
			memset(myBlocks,0,MAX_BLOCKS_DIV8);
		}
	}

	task void executeCmdTask(){
	uint8_t i;
		switch(m_cmd){
			case(prepareCmd):{
				#ifdef STAT
					stat_active_collection=FALSE;
					stat_req_sent=FALSE;
					stat_adv_tx=0;
					stat_req_tx=0;
					stat_data_tx=0;
					stat_cycles=0;
					call StatTimer.stop();
					stat_fixed_rx=0;
					stat_failed_rx=0;
					stat_min_hopcount=0xFF;
					stat_max_hopcount=0;
				#endif
				readyToExecute=TRUE;
				myAppId = INVALID_APP_ID;
				myAppSize = 0;
				myAppBlocks = 0;
				myAppOffset = 0;
				memset(myBlocks,0,MAX_BLOCKS_DIV8);
				blockToSend = INVALID_BLOCK_ID;
				m_blockInBuffer = INVALID_BLOCK_ID;
				retryFlag = 0;
				dataFlag = FALSE;
				advFlag = FALSE;
				reqFlag = FALSE;
				ended = FALSE;
				backoffCount = 0;
				backoffSize = 1;
				cmdSent=0;
				changeStatus(SNP_IDLE,SUCCESS);
			break;
			}
			case (netResetCmd):{
				readyToExecute=FALSE;
				changeStatus(SNP_IDLE,SUCCESS);
			break;
			}
			case (netFormatCmd):{
				if(readyToExecute){
					call BootloaderCommunication.formatFlash();
					reboot();
				}
				else changeStatus(SNP_IDLE,SUCCESS);
			break;
			}
			case (netLoadCmd):{
				if(readyToExecute){
					i=findPartition(myAppId);
					if(i != 0xFF){
						call BootloaderCommunication.loadApplicationPartition(i);
						reboot();
						return;
					}
				}
				changeStatus(SNP_IDLE,SUCCESS);
			break;
			}
			#ifdef STAT
			case (tagCmd):{
				if(readyToExecute){
					memset(tmp_statistics,0,14);
					tmp_statistics [6] = myAppId;
					call LogWrite.append(tmp_statistics, 7*sizeof(uint16_t));
					readyToExecute = FALSE;
					ended = FALSE;
					myAppId = INVALID_APP_ID;
				}
				changeStatus(SNP_IDLE,SUCCESS);
			break;
			}
			#endif
			default:{
			break;
			}
		}		
	}

	uint8_t getNextBlock(uint8_t *bitMask) {
		uint8_t i;
		for (i=0; i<myAppBlocks; i++) {
			if (((bitMask[i>>3])&(1<<(i%8))) && ((myBlocks[i>>3])&(1<<(i%8)))==0) {
				return i+1;
			}
		}
		return 0;
	}

	uint8_t getLastBlock() {
		uint8_t i;
		for (i=0; i<myAppBlocks; i++) {
			if (((myBlocks[i>>3])&(1<<(i%8)))==0) return i;
		}
		return myAppBlocks;
	}

	bool haveMoreBlocks(uint8_t *bitMask){
		uint8_t i;
		uint8_t theirLastBlock=0;
		uint8_t myLastBlock = getLastBlock();
		if (myLastBlock==myAppBlocks) return TRUE;
		for (i=0; i<myAppBlocks; i++) {
				if (((bitMask[i>>3])&(1<<(i%8)))==0) break;
				theirLastBlock++;
			}
		 return (theirLastBlock<myAppBlocks);
	}

	void setBlock(uint8_t block, bool present) {
		block--;
		if (present) myBlocks[block>>3] |= ((uint8_t)1<<(block%8));
		else myBlocks[block>>3] &= ~((uint8_t)1<<(block%8));
	}

	bool blockAvailable(uint8_t block) {
		block--;
		if ((myBlocks[block>>3])&(1<<(block%8))) return TRUE;
		else return FALSE;
	}

	uint8_t findPartition(uint16_t id){
		SynapsePartitionEntry* pe;
		uint8_t n;
		for(n=(EF_PAGE_SIZE/sizeof(SynapsePartitionEntry));n>0;n--){
			pe=call Synapse_PartitionStorage.get_PartitionEntry(n-1);
			if(pe->hashName==id && ((~(pe->flags))&SPE_WORKING)){
					return n-1;
			}
		}
		return 0xFF;
	}

	void SendStatMsg(uint8_t type,uint32_t a,uint32_t b,uint16_t c,uint16_t d,uint8_t e,uint8_t f){
		message_t * m_debugMsg;
		Synapse_Debug_Param_Msg * m_pDebugMsg;
		m_debugMsg = call Synapse_SerialQueue.getMsg();
		m_pDebugMsg = (Synapse_Debug_Param_Msg*)(call SerialPacket.getPayload(m_debugMsg, 0));
		m_pDebugMsg->type = type;
		m_pDebugMsg->p32_1 = a;
		m_pDebugMsg->p32_2 = b;
		m_pDebugMsg->p16_1 = c;
		m_pDebugMsg->p16_2 = d;
		m_pDebugMsg->p8_1 = e;
		m_pDebugMsg->p8_2 = f;
		call Synapse_SerialQueue.send( m_debugMsg, sizeof(Synapse_Debug_Param_Msg) );
	}


	void changeStatus(uint8_t newStatus,error_t error){
		status=newStatus;
		call Leds.led0Off();
		call Leds.led1Off();
		call Leds.led2Off();
		if(ended) call Leds.set(7);
		switch(status){
			case SNP_IDLE:{
				if (error==FAIL) call Leds.led0On();
				else call Leds.led1On();
				if (readyToExecute) call Leds.led2On();
			break;
			}
			case SNP_ADV:{
			break;
			}
			case SNP_REQ:{
			break;
			}
			case SNP_DATA:{
			break;
			}
			case SNP_CMD:{
				call Leds.led0On();
				call Leds.led1On();
			break;
			}
			default:
			break;
		}
	}
	
	void pingManage(message_t* pingAMMessage){
#ifdef DEBUG_SYNAPSE
		bool blinkFlag=FALSE;
#endif
		SynapsePartitionEntry* partitionEntry;
		Synapse_Ping* ping_msg;
		ping_msg=call PingPacket.getPayload(pingAMMessage, 0);
		if (ping_msg->dest==TOS_NODE_ID && (((ping_msg->pingFlags)&PING_REPLY_FLAG)!=PING_REPLY_FLAG)){
			if (ping_msg->partNum <= (EF_PAGE_SIZE/sizeof(SynapsePartitionEntry))) {
				partitionEntry=call Synapse_PartitionStorage.get_PartitionEntry(ping_msg->partNum);
				ping_msg->imgId=partitionEntry->hashName;
				ping_msg->imgFlags=partitionEntry->flags;
				ping_msg->imgSize=partitionEntry->size;
				ping_msg->imgOffset=partitionEntry->startAddr;
			} else {
				ping_msg->imgId=(EF_PAGE_SIZE/sizeof(SynapsePartitionEntry));
				ping_msg->pingFlags|=PING_OVERFLOW_FLAG;
				ping_msg->imgSize=call get_NumOfImg();
				ping_msg->imgOffset=0;
			}
			ping_msg->dest=pingSrc;
			ping_msg->status=status;
			ping_msg->pingFlags=(ping_msg->pingFlags|PING_REPLY_FLAG);
#ifdef DEBUG_SYNAPSE
			blinkFlag=TRUE;
#endif
			if(readyToExecute) ping_msg->pingFlags=(ping_msg->pingFlags|PING_PREPARED_FLAG);
		}
		if((((ping_msg->pingFlags)&PING_REPLY_FLAG)==PING_REPLY_FLAG) && serialConnected){
			SendStatMsg(pingRepMsg,ping_msg->imgSize,0,ping_msg->imgId,ping_msg->imgFlags,ping_msg->pingFlags,ping_msg->status);
#ifdef DEBUG_SYNAPSE
			if (blinkFlag==TRUE) call DumpToLeds.dump4bitR(0x7, 30);
#endif
			return;
		}
		call PingSend.send(ping_msg->dest,pingAMMessage,sizeof(Synapse_Ping));
#ifdef DEBUG_SYNAPSE
		if (blinkFlag==TRUE) call DumpToLeds.dump4bitR(0x7, 30);
#endif
		return;
	}

	void syncOnLastSfdRx(uint32_t sfdRxTime, uint32_t RxEventTime, uint32_t baseTime){
		uint32_t delay = RxEventTime-sfdRxTime;
		if (delay > baseTime) {
			SendStatMsg(0xFF,delay-baseTime,0,0,0,0,0);
			return;
		}
		call Timer.startOneShot(baseTime-delay+2);
	}

	command error_t Init.init(){
#ifndef LOADABLE
			if (call BootloaderCommunication.writeNodeId(TOS_NODE_ID) != SUCCESS) call Leds.led0On() /*DumpToLeds.dump4bitR(0x4,20)*/;
#else
			am_addr_t nodeId;
			nodeId=call BootloaderCommunication.readNodeId();
			atomic {
				TOS_NODE_ID = nodeId;
				call setAmAddress(nodeId);
			}
			call CC2420Config.setShortAddr(nodeId);
			call CC2420Config.sync();
#endif
		return SUCCESS;
	}

	event void SerialControl.stopDone(error_t err){
	}

	event void AMControl.stopDone(error_t err){
		call SerialControl.stop();
	}

	event message_t* SerialReceive.receive(message_t* msg, void* payload, uint8_t len)
	{
		Synapse_Cmd_Msg* m_cmd_payload;
		baseStation=TRUE;
		cmdSent=0;
		call Leds.led0On();
#ifdef STAT
			stat_active_collection=FALSE;
			stat_req_sent=FALSE;
			stat_adv_tx=0;
			stat_req_tx=0;
			stat_data_tx=0;
			stat_cycles=0;
			call StatTimer.stop();
			stat_fixed_rx=0;
			stat_failed_rx=0;
			stat_min_hopcount=0xFF;
			stat_max_hopcount=0;
#endif
#ifdef DEBUG_SYNAPSE
		SendStatMsg(DBG_TIMES, adv_time, req_time, total_data_transmission_time, adv_slot, 0, 0);
#endif
		if ((status==SNP_IDLE) || (((Synapse_serialCommand*)payload)->cmd == prepareCmd)){
			m_cmd=((Synapse_serialCommand*)payload)->cmd;
			if(m_cmd==transferCmd || m_cmd==netLoadCmd){
				myAppId=((Synapse_serialCommand*)payload)->arg;
			}
			else myAppId=INVALID_APP_ID;
			switch(m_cmd){
				case (transferCmd):{
					call Synapse_PartitionStorage.mountId(((Synapse_serialCommand*)payload)->arg);
#ifdef STAT
						stat_active_collection=TRUE;
						call StatTimer.startPeriodic(SLOT_NUM*2*adv_slot + GUARD_INTERVAL*6 + TOTAL_DATA_TRANSMISSION_TIME + MAX_DECODE_TIME);
#endif
					flashStatus = FLASH_MOUNTING;
					changeStatus(SNP_CMD,SUCCESS);
				break;
				}
#ifdef STAT
				case (tagCmd):
					readyToExecute = TRUE;
					myAppId=((Synapse_serialCommand*)payload)->arg;
#endif
				case (prepareCmd):
				case (netLoadCmd):
				case (netResetCmd):
				case (netFormatCmd):{
					changeStatus(SNP_CMD,SUCCESS);
					myAppSize = 0;
					myAppBlocks = 0;
					myAppOffset = 0x0000;
					memset(myBlocks,0,MAX_BLOCKS_DIV8);
					m_blockInBuffer = INVALID_BLOCK_ID;
					blockToSend = INVALID_BLOCK_ID;
					retryFlag = 0;
					dataFlag = FALSE;
					advFlag = FALSE;
					reqFlag = FALSE;
					backoffCount = 0;
					backoffSize = 1;
					m_cmd_payload = (Synapse_Cmd_Msg*)(call CmdPacket.getPayload(&m_cmd_msg, 0));
					m_cmd_payload->cmd=m_cmd;
					m_cmd_payload->appId=myAppId;
					call Timer.stop();
					call TxTimer.startOneShot(RAND_SLOT*adv_slot);
				break;
				}
				default:{
				break;
				}
			}
		}
		return msg;
	}

	event message_t* CmdReceive.receive(message_t* msg, void* payload, uint8_t len) {
		Synapse_Cmd_Msg *cmd_payload;
		if ((status==SNP_IDLE) || ( (((Synapse_Cmd_Msg*)payload)->cmd == prepareCmd) && (readyToExecute==FALSE) ) ){
			if (((Synapse_Cmd_Msg*)payload)->cmd==m_cmd) return msg;
			switch (((Synapse_Cmd_Msg*)payload)->cmd){
			case (prepareCmd):
				changeStatus(SNP_CMD,SUCCESS);
				m_cmd = ((Synapse_Cmd_Msg*)payload)->cmd;
				cmdSent = 0;
				cmd_payload = (Synapse_Cmd_Msg*)(call CmdPacket.getPayload(&m_cmd_msg, 0));
				cmd_payload->cmd=m_cmd;
				call Timer.stop();
				call TxTimer.startOneShot(RAND_SLOT*adv_slot);
			break;
#ifdef STAT
			case (tagCmd):
#endif
			case (netLoadCmd):
			case (netResetCmd):
			case (netFormatCmd):
				if (status!=SNP_IDLE) break;
				changeStatus(SNP_CMD, SUCCESS);
				m_cmd = ((Synapse_Cmd_Msg*)payload)->cmd;
				myAppId = ((Synapse_Cmd_Msg *)payload)->appId;
				cmd_payload = (Synapse_Cmd_Msg*)(call CmdPacket.getPayload(&m_cmd_msg, 0));
				cmd_payload->cmd=m_cmd;
				cmd_payload->appId=myAppId;
				cmdSent = 0;
				call TxTimer.startOneShot(RAND_SLOT*adv_slot);
			break;
			default:
			break;
			}
		}
		return msg;
	}

	event message_t* AdvReceive.receive(message_t* msg, void* payload, uint8_t len) {
			uint32_t sfdRxTime;
			uint32_t now; 
			Synapse_Req_Msg *m_req_payload;
			m_req_payload = (Synapse_Req_Msg*)(call ReqPacket.getPayload(&m_req_msg, 0));
			if (call TimeSyncAdvPacket.isValid(msg)) sfdRxTime = call TimeSyncAdvPacket.eventTime(msg);
			else sfdRxTime = call Timer.getNow() - 5; //fallback ipothesis if TimeSync event is not valid
			now = call Timer.getNow();
			if (readyToExecute || (flashStatus != FLASH_IDLE) || sending) return msg;
#ifdef DEBUG_SYNAPSE
	 		if (baseStation) SendStatMsg(DBG_ADV_RCV,0,0,mySource,0,((Synapse_Adv_Msg*)payload)->blocks,0);
			SendStatMsg(0xFF,myHopCount,myAppBlocks,advCheck,reqFlag,status,1);
#endif
			switch(status){
			case(SNP_IDLE): {
#ifdef STAT
					stat_active_collection=TRUE;
					stat_cycles = ((Synapse_Adv_Msg*)payload)->cycles;
					call StatTimer.startPeriodic(SLOT_NUM*2*adv_slot + GUARD_INTERVAL*6 + TOTAL_DATA_TRANSMISSION_TIME + MAX_DECODE_TIME);
#endif
				changeStatus(SNP_ADV, SUCCESS);
 				SendStatMsg(0xFF,0,0,0,0,0,2);
				flashStatus = FLASH_CREATING;
				myHopCount=(((Synapse_Adv_Msg*)payload)->hopCount)+1;
				mySource = (am_addr_t)(call AdvAMPacket.source(msg));
				myAppId = ((Synapse_Adv_Msg*)payload)->appId;
				syncOnLastSfdRx(sfdRxTime,now,(((Synapse_Adv_Msg*)payload)->slot)*adv_slot+GUARD_INTERVAL);
#ifdef DISSEMINATE_STORED_IMAGES
				if(findPartition(myAppId) != 0XFF){
					flashStatus = FLASH_MOUNTING;
					call Synapse_PartitionStorage.mountId(myAppId);
					return msg;
				}
#endif
				if ((((Synapse_Adv_Msg*)payload))->slot >= (SLOT_NUM/2)) reqDelayedFlag = FALSE;
				else reqDelayedFlag = TRUE;
				reqTxSlot = RAND_SLOT + SLOT_NUM/2*reqDelayedFlag;
// 				SendStatMsg(0xFF,sfdRxTime,now,0,0,0,3);
				m_req_payload->blocks =  1;
				call Synapse_PartitionStorage.createPartition(((Synapse_Adv_Msg*)payload)->appId, 
																((Synapse_Adv_Msg*)payload)->size,
																((Synapse_Adv_Msg*)payload)->progStartOffs);
// 				SendStatMsg(0xFF,0,0,0,0,0,4);
			break;
			}
			case(SNP_ADV):{
				if(myAppId==INVALID_APP_ID){
					flashStatus = FLASH_CREATING;
					myHopCount=(((Synapse_Adv_Msg*)payload)->hopCount)+1;
					mySource = (am_addr_t)(call AdvAMPacket.source(msg));
					myAppId = ((Synapse_Adv_Msg*)payload)->appId;
					if ((((Synapse_Adv_Msg*)payload))->slot >= (SLOT_NUM/2)) reqDelayedFlag = FALSE;
					else reqDelayedFlag = TRUE;
					reqTxSlot = RAND_SLOT + SLOT_NUM/2*reqDelayedFlag;
					syncOnLastSfdRx(sfdRxTime,now,(((Synapse_Adv_Msg*)payload)->slot)*adv_slot+GUARD_INTERVAL);
					call Synapse_PartitionStorage.createPartition(((Synapse_Adv_Msg*)payload)->appId, 
																	((Synapse_Adv_Msg*)payload)->size,
																	((Synapse_Adv_Msg*)payload)->progStartOffs);
					return msg;
				}
				if (advFlag) {
					if ((((Synapse_Adv_Msg*)payload))->slot >= (SLOT_NUM/2)) {  //if we received a prio ADV
						if (!advDelayedFlag)  advFlag = FALSE;											//suppress any prio ADV
					}
					else if (advDelayedFlag){ //if we received a late adv and their have more blocks than us do then suppress any other ADV
					if((haveMoreBlocks((uint8_t *)(((Synapse_Adv_Msg*)payload)->blocks)))) advFlag = FALSE;	
					}
				}
				if((((Synapse_Adv_Msg*)payload)->appId==myAppId) && (!reqFlag) /*&& (!advCheck)*/) {
					if ((((Synapse_Adv_Msg*)payload))->slot >= (SLOT_NUM/2) && (retryFlag)) return msg;
					if (getNextBlock((uint8_t *)(((Synapse_Adv_Msg*)payload)->blocks))) {
						if (!advCheck) syncOnLastSfdRx(sfdRxTime,now,(((Synapse_Adv_Msg*)payload)->slot)*adv_slot+GUARD_INTERVAL);
						mySource = (am_addr_t)(call AdvAMPacket.source(msg));
						if ((((Synapse_Adv_Msg*)payload))->slot >= (SLOT_NUM/2)) reqDelayedFlag = FALSE;
						else reqDelayedFlag = TRUE;
						reqTxSlot = RAND_SLOT + SLOT_NUM/2*reqDelayedFlag;
						myHopCount=(((Synapse_Adv_Msg*)payload)->hopCount)+1;
						if (!retryFlag) m_req_payload->blocks = getNextBlock((uint8_t *)(((Synapse_Adv_Msg*)payload)->blocks));
						else m_req_payload->blocks = retryFlag;
						call Synapse_Radio.initializeReceive(m_req_payload->blocks,retryFlag>0);
						if(flashStatus==FLASH_IDLE && (getLastBlock() != myAppBlocks)){	//else wait until writeDone event
							reqFlag = TRUE;
#ifdef DEBUG_SYNAPSE
 								SendStatMsg(DBG_INI_REC,0,0,0,0,nextBlock,retryFlag);
#endif 								
						}
					}
				}
				if((!call Timer.isRunning()) || ((((Synapse_Adv_Msg*)payload)->hopCount<myHopCount) 
					&& (!reqFlag) && (!advCheck))) { //in any case sync on lower hop count
#ifdef DEBUG_SYNAPSE
 						SendStatMsg(0xFF,0,0,0,0,0,9);
#endif
					syncOnLastSfdRx(sfdRxTime,now,(((Synapse_Adv_Msg*)payload)->slot)*adv_slot+GUARD_INTERVAL);
				}
			break;
			}
			case SNP_REQ:
			case SNP_DATA:{
#ifdef DEBUG_SYNAPSE
 				SendStatMsg(DBG_WRONG_ADV,0,0,(am_addr_t)(call AdvAMPacket.source(msg)),0,((Synapse_Adv_Msg*)payload)->slot,status);
#endif
				if((!call Timer.isRunning()) || ((((Synapse_Adv_Msg*)payload)->hopCount<myHopCount) && 
					(!reqFlag) && (!advCheck))){ //in any case sync on lower level FIXME
					changeStatus(SNP_ADV,SUCCESS);
					syncOnLastSfdRx(sfdRxTime,now,(((Synapse_Adv_Msg*)payload)->slot)*adv_slot+GUARD_INTERVAL);
					if(((Synapse_Adv_Msg*)payload)->appId==myAppId) {
						if (getNextBlock((uint8_t *)(((Synapse_Adv_Msg*)payload)->blocks))) {
							call TxTimer.stop();
							mySource = (am_addr_t)(call AdvAMPacket.source(msg));
							if ((((Synapse_Adv_Msg*)payload))->slot >= (SLOT_NUM/2)) reqDelayedFlag = FALSE;
							else reqDelayedFlag = TRUE;
							reqTxSlot = RAND_SLOT + SLOT_NUM/2*reqDelayedFlag;
							myHopCount=(((Synapse_Adv_Msg*)payload)->hopCount)+1;
							if (!retryFlag) m_req_payload->blocks = getNextBlock((uint8_t *)(((Synapse_Adv_Msg*)payload)->blocks));
							else m_req_payload->blocks = retryFlag;
							call Synapse_Radio.initializeReceive(m_req_payload->blocks,retryFlag>0);
							if(flashStatus==FLASH_IDLE && (getLastBlock() != myAppBlocks)){	//else wait until writeDone event
								reqFlag = TRUE;
// 								SendStatMsg(DBG_INI_REC,0,0,0,0,nextBlock,retryFlag);
							}
						}
					}
				}
			break;
			}
			default:{
#ifdef DEBUG_SYNAPSE
 				SendStatMsg(0xFF,0,0,0,0,status,11);
#endif
			break;
			}
		}
#ifdef DEBUG_SYNAPSE
 		SendStatMsg(0xFF,0,0,0,0,0,12);
#endif
		return msg;
	}

	event message_t* PingReceive.receive(message_t* msg, void* payload, uint8_t len){
		pingSrc=call PingAMPacket.source(msg);
		pingManage(msg);
		return msg;
	}
	
	event message_t* SerialPingReceive.receive(message_t* msg, void* payload, uint8_t len){
		serialConnected=TRUE;
		pingManage(msg);
		return msg;
	}
	
	event message_t* ReqReceive.receive(message_t* msg, void* payload, uint8_t len){
		if (readyToExecute || (flashStatus != FLASH_IDLE) || sending) return msg;
#ifdef DEBUG_SYNAPSE
 		if (myHopCount==0) SendStatMsg(DBG_REQ_RCV,0,0,((Synapse_Req_Msg *)payload)->s_addr,
 										0,((Synapse_Req_Msg *)payload)->blocks,status);
#endif
		switch(status){
			case SNP_REQ:{
				if(reqFlag){
					reqFlag = FALSE;
					if(!blockAvailable(((Synapse_Req_Msg *)payload)->blocks)) //check if I need it
						call Synapse_Radio.initializeReceive(((Synapse_Req_Msg *)payload)->blocks,(((Synapse_Req_Msg *)payload)->blocks)==retryFlag);
				}
				if (((Synapse_Req_Msg *)payload)->s_addr == TOS_NODE_ID) {
#ifdef DEBUG_SYNAPSE					
					SendStatMsg(DBG_REQ_RCV,0,0,((Synapse_Req_Msg *)payload)->s_addr,0,((Synapse_Req_Msg *)payload)->blocks,status);
#endif
					dataFlag = TRUE;
					advDelayedFlag = TRUE;
					if (blockToSend < (((Synapse_Req_Msg *)payload)->blocks)) {
						if (blockAvailable(((Synapse_Req_Msg *)payload)->blocks)) blockToSend=((Synapse_Req_Msg *)payload)->blocks;
						else dataFlag = FALSE;
					}
				}
			break;
			}
			default:{
#ifdef DEBUG_SYNAPSE				
					if (((Synapse_Req_Msg *)payload)->s_addr == TOS_NODE_ID)
					SendStatMsg(DBG_WRONG_REQ,0,0,(am_addr_t)(call ReqAMPacket.source(msg)),((Synapse_Req_Msg*)payload)->s_addr,0,status);
#endif
			break;
			}
		}
		return msg;
	}

	event void ReqSend.sendDone(message_t* msg, error_t err)
	{
#ifdef STAT
		if (stat_active_collection) {
			stat_req_tx++;
			stat_req_sent = TRUE;
		}
#endif
	}

	event void PingSend.sendDone(message_t* msg, error_t err){}

	event void CmdSend.sendDone(message_t* msg, error_t err) 
	{
		cmdSent++;
		if (cmdSent<MAX_CMD) call TxTimer.startOneShot(RAND_SLOT*adv_slot);
		else {
#ifdef STAT
			if(!baseStation || m_cmd==tagCmd) post executeCmdTask();
#else
			if(!baseStation) post executeCmdTask();
#endif
			else changeStatus(SNP_IDLE, SUCCESS);
		}
	}

	event void AdvSend.sendDone(message_t* msg, error_t err)
	{
		Synapse_Adv_Msg *tmp = (Synapse_Adv_Msg*)(call AdvPacket.getPayload(&m_adv_msg, 0));
#ifdef STAT
			if (stat_active_collection) stat_adv_tx++;
#endif
		if ((myHopCount==0) && (!advDelayedFlag)) advCycles=0;
		if (backoffSize < MAX_BACKOFF) backoffSize *= 2;
#ifdef DEBUG_SYNAPSE
		SendStatMsg(DBG_ADV_SNT,tmp->appId,tmp->slot,tmp->hopCount,tmp->size,advTxSlot,getLastBlock());
#endif
		blockToSend=0;
		advCheck = TRUE;
	}

#ifdef LOADABLE
	event void CC2420Config.syncDone(error_t error){
	}	
#endif

	event void Synapse_Radio.sendBlockDone(error_t err){
		sending=FALSE;
		lastSent = blockToSend;
#ifdef STAT
			if (stat_active_collection) {
				stat_data_tx++;
				if (lastSent == myAppBlocks) {
					stat_active_collection=FALSE;
					call StatTimer.stop();
					tmp_statistics [0] = stat_adv_tx;
					tmp_statistics [1] = stat_req_tx;
					tmp_statistics [2] = stat_data_tx;
					tmp_statistics [3] = stat_cycles;
					tmp_statistics [4] = stat_fixed_rx;
					tmp_statistics [5] = stat_failed_rx;
					tmp_statistics [6] = stat_min_hopcount;
					tmp_statistics [6] |= stat_max_hopcount<<8;
					call LogWrite.append(tmp_statistics, 7*sizeof(uint16_t));
				}
			}
#endif
#ifdef DEBUG_SYNAPSE
		SendStatMsg(DBG_DATA_SNT,0,0,0,0,0,lastSent);
#endif
		dataFlag=FALSE;
		if (backoffSize >= 2) backoffSize /= 2;
		else backoffSize = 1;
	}

	event void Synapse_Radio.receiveBlockDone(uint8_t *PASS buf, uint16_t size, uint16_t blockId, error_t error ){
		uint16_t toWrite;
		switch(status){
			case SNP_DATA:{
				if (error != SUCCESS) {  //means decode failure
					if (size>0) {
#ifdef STAT
						if (stat_active_collection) {
							if (stat_req_sent) stat_failed_rx++;
							stat_req_sent = FALSE;
						}
#endif
						retryFlag = blockId;
						advFlag = FALSE;
						m_blockInBuffer = INVALID_BLOCK_ID;
#ifdef DEBUG_SYNAPSE
 						SendStatMsg(DBG_DATA_NAK,0,retryFlag,size,backoffCount,blockId,backoffSize);
#endif
					}
					else {
						retryFlag = 0;
#ifdef DEBUG_SYNAPSE
 						SendStatMsg(DBG_DATA_NAK,0,retryFlag,size,backoffCount,blockId,backoffSize);
#endif
					}
					return;
				}
				else {
#ifdef STAT
					if (stat_active_collection)	if (retryFlag) stat_fixed_rx++;
#endif
					if(flashStatus != FLASH_IDLE) return;
#ifdef DEBUG_SYNAPSE
					SendStatMsg(DBG_DATA_RCV,0,0,size,0,blockId,0);
#endif
					flashStatus = FLASH_WRITING;
#ifdef CHECK_BLOCK_CRC
#warning CHECK_BLOCK_CRC enabled
// 					the CRC check takes approx 20 ms
					if(call Crc.crc16(m_buffer,BLOCK_PAYLOAD_SIZE) != ((uint16_t*)&(m_buffer[BLOCK_PAYLOAD_SIZE]))[0]){
						retryFlag = 0;
						m_blockInBuffer = INVALID_BLOCK_ID;
						flashStatus = FLASH_IDLE;
#ifdef DEBUG_SYNAPSE
						SendStatMsg(DBG_CRC_FAILED,0,0,0,0,0,0);
						call DumpToLeds.dump8bitR(0x44,30);
#endif
						return;
					}						
#endif
					call Leds.set(myHopCount);
#ifdef STAT
					if (stat_active_collection) {
						if (myHopCount<stat_min_hopcount) stat_min_hopcount = myHopCount;
						if (myHopCount>stat_max_hopcount) stat_max_hopcount = myHopCount;
					}
#endif
					m_blockInBuffer = blockId;
					toWrite=MIN(m_bufferSize-CRC_SIZE, 1 + myAppSize - (((uint32_t)blockId-1)*BLOCK_PAYLOAD_SIZE));
					if (call BlockWrite.write(((uint32_t)blockId-1)*BLOCK_PAYLOAD_SIZE, m_buffer, toWrite)!=SUCCESS){
						flashStatus = FLASH_IDLE;
					}
					else setBlock(blockId,1);
				}
					
			break;
			}
			default:{
			break;
			}
		}
	}

#ifdef STAT
	event void StatTimer.fired() {
		if (stat_active_collection) stat_cycles++;
	}
#endif

	event void Timer.fired()	// sync timer
	{
		uint16_t toRead, j;
		switch(status){
			case SNP_ADV:{			// ADV time slot ended
				call Timer.startOneShot(req_slot*(SLOT_NUM)+2*GUARD_INTERVAL);
				if (reqFlag==TRUE) {
					call TxTimer.startOneShot(reqTxSlot*req_slot+GUARD_INTERVAL);
				}
				changeStatus(SNP_REQ, SUCCESS);
			break;
			}
			case SNP_REQ:{			// REQ time slot ended
				call Timer.startOneShot(total_data_transmission_time + decode_time + 2*GUARD_INTERVAL);
				if(!dataFlag) {
					if (advCheck) backoffCount = (call Random.rand16())%backoffSize;
					call Synapse_Radio.receiveBlock();
				}
				else{
					toRead = MIN(m_bufferSize-CRC_SIZE, 1 + myAppSize - (((uint32_t)blockToSend-1)*BLOCK_PAYLOAD_SIZE));
					if(toRead<m_bufferSize){
						for(j=toRead;j<m_bufferSize;j++) ((uint8_t*)m_buffer)[j]=0;
					}
					call Synapse_Radio.initializeSend(blockToSend,seeds[(call Random.rand16())%NUM_OF_SEEDS]);	
					if (1 || m_blockInBuffer!=blockToSend) {
						if (call BlockRead.read(((uint32_t)blockToSend-1)*BLOCK_PAYLOAD_SIZE, m_buffer, toRead)==SUCCESS) {
							flashStatus = FLASH_READING;
// 							SendStatMsg(DBG_FLASH_READING,(uint16_t)m_buffer,0,0,((uint32_t)blockToSend-1)*BLOCK_PAYLOAD_SIZE,0,0);
						}
#ifdef DEBUG_SYNAPSE						 
						else while(1) call DumpToLeds.dump16bitR(0x5151, 10); // TODO remount after a flash failure
#endif
					} else {
						if (dataFlag == TRUE) call TxTimer.startOneShot(MINIMUM_SKEW+GUARD_INTERVAL);
					}
				}
				advCheck = FALSE;
				reqFlag=FALSE; 			//just in case reset it at every cycle
				changeStatus(SNP_DATA, SUCCESS);
			break;
			}
			case SNP_DATA:{		// DATA time slot ended
				call Timer.startOneShot(adv_slot*(SLOT_NUM)+2*GUARD_INTERVAL);
				if ((getLastBlock() >= DISSEMINATION_BLOCK_TRESHOLD) && (!retryFlag) && ((myHopCount==0) || (backoffCount==0))) {
					advFlag = TRUE;
					advTxSlot = RAND_SLOT + SLOT_NUM/2*advDelayedFlag;
					call TxTimer.startOneShot(advTxSlot*adv_slot+GUARD_INTERVAL);
				}
				if (backoffCount > 0) backoffCount--;
				changeStatus(SNP_ADV, SUCCESS);
			break;
			}
			default:{
			break;
			}
		}
	}

	event void TxTimer.fired()		// transmission timer
	{
	Synapse_Adv_Msg *m_adv_payload;
	Synapse_Req_Msg *m_req_payload;

		switch(status){
			case SNP_ADV:{
				if (advFlag==TRUE) {
					m_adv_payload = (Synapse_Adv_Msg*)(call AdvPacket.getPayload(&m_adv_msg, 0));
					m_adv_payload->appId=myAppId;
					m_adv_payload->size=myAppSize;
					m_adv_payload->progStartOffs=myAppOffset;
					m_adv_payload->hopCount = myHopCount;
#ifdef STAT
					m_adv_payload->cycles = stat_cycles;
#endif
					memcpy(&(m_adv_payload->blocks),myBlocks,MAX_BLOCKS_DIV8*sizeof(uint8_t));
					m_adv_payload->slot=((SLOT_NUM-1)-advTxSlot);
					advDelayedFlag = TRUE;
					call AdvSend.send(AM_BROADCAST_ADDR, &m_adv_msg , sizeof(Synapse_Adv_Msg), call Timer.getNow());
				}
			break;
			}
			case SNP_REQ:{
				if (reqFlag==TRUE) {
					m_req_payload = (Synapse_Req_Msg*)(call ReqPacket.getPayload(&m_req_msg, 0));
					m_req_payload->s_addr = mySource;
					reqFlag=FALSE;
					call ReqSend.send(AM_BROADCAST_ADDR, &m_req_msg , sizeof(Synapse_Req_Msg));
#ifdef DEBUG_SYNAPSE
					SendStatMsg(DBG_REQ_SNT,0,0,0,mySource,reqTxSlot,m_req_payload->blocks);
#endif
				}
			break;
			}
			case SNP_DATA:{
 				if ((!sending)&&(dataFlag)&&(m_blockInBuffer==blockToSend)&&(flashStatus==FLASH_IDLE)) {
 					sending=TRUE;
					call Synapse_Radio.sendBlock();
				}
			break;
			}
			case SNP_CMD: {
				call CmdSend.send(AM_BROADCAST_ADDR, &m_cmd_msg , sizeof(Synapse_Cmd_Msg));
			break;	
			}
			default:{
			break;
			}
		}
	}
	
	event void Synapse_PartitionStorage.mounted( uint16_t id, uint16_t pos, uint16_t size,uint16_t offset, error_t error ){
		uint8_t i;
		Synapse_Adv_Msg* m_adv_payload;
		flashStatus = FLASH_IDLE;
		if(error==SUCCESS) {
			myAppId = id;
			myAppSize = size;
			myAppOffset = offset;
			myAppBlocks = CEIL(size,BLOCK_PAYLOAD_SIZE);
			if(baseStation) myHopCount=0;
			for (i=0; i<myAppBlocks; i++) setBlock(i+1,1);
			m_adv_payload = (Synapse_Adv_Msg*)(call AdvPacket.getPayload(&m_adv_msg, 0));
			m_adv_payload->appId=myAppId;
			m_adv_payload->size=size;
			m_adv_payload->progStartOffs=offset;
#ifdef STAT
			m_adv_payload->cycles = stat_cycles;
#endif
			memcpy(&(m_adv_payload->blocks),myBlocks,MAX_BLOCKS_DIV8*sizeof(uint8_t));
			advTxSlot=1;
			m_adv_payload->slot=(SLOT_NUM-advTxSlot);
			changeStatus(SNP_ADV, SUCCESS);
			advFlag=TRUE;
			if(baseStation){
				call AdvSend.send(AM_BROADCAST_ADDR, &m_adv_msg , sizeof(Synapse_Adv_Msg), call Timer.getNow());
				call Timer.startOneShot(GUARD_INTERVAL + (SLOT_NUM-advTxSlot)*adv_slot);
			}
		} else {
			changeStatus(SNP_IDLE, FAIL);
		}
	}

	event void BlockRead.readDone(storage_addr_t addr, void* buf, storage_len_t len, error_t error){
		if(error==SUCCESS) {
#ifdef CHECK_BLOCK_CRC
			((uint16_t*)&(m_buffer[BLOCK_PAYLOAD_SIZE]))[0] = call Crc.crc16(m_buffer,BLOCK_PAYLOAD_SIZE);
#endif
			m_blockInBuffer = blockToSend;
			flashStatus = FLASH_IDLE;
			call TxTimer.startOneShot(MINIMUM_SKEW+GUARD_INTERVAL);
		} 
#ifdef DEBUG_SYNAPSE		
		else call DumpToLeds.dump8bitR(0x15,50);
#endif
	}

	event void Synapse_PartitionStorage.created( uint16_t id, uint16_t pos, uint16_t size, uint16_t offset, error_t error ){
		if(error != SUCCESS){
			myAppId = INVALID_APP_ID;			
			changeStatus(SNP_IDLE,FAIL);
		}
		myAppSize = size;
		myAppOffset = offset;
		myAppBlocks = CEIL(size,BLOCK_PAYLOAD_SIZE);
		flashStatus = FLASH_IDLE;
		memset(myBlocks,0,MAX_BLOCKS_DIV8);
		call Synapse_Radio.initializeReceive(1,0);
// 		SendStatMsg(0xFF,error==SUCCESS,0,myAppSize,myAppBlocks,0xFF,66);
		if (status == SNP_ADV) {
			reqFlag = TRUE;
// 			SendStatMsg(0xFF,call Timer.getNow(),0,0,0,0,5);
		}
	}

	event void Synapse_PartitionStorage.flagsChanged(error_t error){
#ifdef STAT
		if (stat_active_collection) {
			stat_active_collection=FALSE;
			call StatTimer.stop();
			tmp_statistics [0] = stat_adv_tx;
			tmp_statistics [1] = stat_req_tx;
			tmp_statistics [2] = stat_data_tx;
			tmp_statistics [3] = stat_cycles;
			tmp_statistics [4] = stat_fixed_rx;
			tmp_statistics [5] = stat_failed_rx;
			tmp_statistics [6] = stat_min_hopcount;
			tmp_statistics [6] |= stat_max_hopcount<<8;
			call LogWrite.append(tmp_statistics, 7*sizeof(uint16_t));
			SendStatMsg(0xFE,stat_adv_tx,stat_req_tx,stat_data_tx,stat_cycles,stat_min_hopcount,stat_max_hopcount);
			SendStatMsg(0xFD,0,0,stat_fixed_rx,stat_failed_rx,0,0);
		}
#endif
		ended = TRUE;
		call Leds.set(7);
#ifdef DEBUG_SYNAPSE
 		SendStatMsg(DBG_FLASH_WRITTEN,0xff,0,0,0xff,0,0);
#endif
	}

	event void BlockRead.computeCrcDone(storage_addr_t addr, storage_len_t len, uint16_t crc, error_t error){ 
	}
	
	event void BlockWrite.writeDone(storage_addr_t addr, void* buf, storage_len_t len, error_t error){
		uint8_t temp_last_block = getLastBlock();
		flashStatus = FLASH_IDLE;
		if (error != SUCCESS){
			setBlock(m_blockInBuffer,0);
			return;
		}
		call Synapse_Radio.initializeReceive(temp_last_block+1,FALSE);
#ifdef DEBUG_SYNAPSE
 		SendStatMsg(DBG_INI_REC,0,0,0,0,temp_last_block+1,retryFlag);
 		SendStatMsg(DBG_FLASH_WRITTEN,myAppBlocks,0,0,addr,0,0);
#endif		
	if ((temp_last_block >= DISSEMINATION_BLOCK_TRESHOLD) && (temp_last_block <= myAppBlocks)) {
			advFlag = TRUE;
			retryFlag = 0;
			advDelayedFlag = FALSE;
			backoffSize = 1;
			backoffCount = 0;
			advTxSlot = RAND_SLOT + SLOT_NUM/2*advDelayedFlag;
		}
		if(temp_last_block == myAppBlocks){
			call Synapse_PartitionStorage.setWorking();
// 			SendStatMsg(DBG_FLASH_WRITTEN,0xff,0,0,0xff,0,setworking);
		}
	}

	event void BlockWrite.eraseDone(error_t error){
	}

	event void BlockWrite.syncDone(error_t error){
	}

#ifdef STAT
	event void LogWrite.appendDone(void* buf, storage_len_t len, bool recordsLost,error_t error){
		if(error != SUCCESS) call LogWrite.append(buf, len);
		else call LogWrite.sync();
	}
	event void LogWrite.eraseDone(error_t error){}
	event void LogWrite.syncDone(error_t error){
		if(error != SUCCESS) call LogWrite.sync();
	}
#endif

}
