/*
 * Copyright (c) 2008 Department of Information Engineering, University of Padova, Italy
 * Contributors: Giovanni Zanca, Nicola Bui, Riccardo Crepaldi and Michele Rossi
 * All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 *       copyright notice, this list of conditions and the following
 *       disclaimer in the documentation and/or other materials provided
 *       with the distribution.
 *     * The name of the author may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "Synapse.h"



#define SET_BIT(mem, idx)       ( (mem) |= 1<<(idx) )
#define CLEAR_BIT(mem, idx)     ( (mem) &= ~(1<<(idx)) )
#define TEST_BIT(mem, idx)      ( (mem) & 1<<(idx) )

#define SYMBOL_SIZE             SNP_DATA_SIZE  // in bytes

#define DECODER_SNP_BLOCK_SIZE      SYMBOL_SIZE * RECEIVER_SNP_BLOCK_SIZE
#define BIT_MATRIX_SIZE         SYMBOLS_PER_BLOCK_DIV8*RECEIVER_SNP_BLOCK_SIZE

module Synapse_CodecP
{
	provides {
		interface Synapse_Codec as Codec;
	}
	uses {
		interface Synapse_Random as RandPcks;
#ifdef DEBUG_SYNAPSE_RADIO_CODEC
		interface DumpToLeds;
#endif
	}
}

implementation
{
#if SYMBOLS_PER_BLOCK == 32
#warning SYMBOLS_PER_BLOCK 32
	// Rossi distribution 32
	static const uint16_t sumPhi[SYMBOLS_PER_BLOCK] ={
	0x19BC, 0x3FF0, 0x595D, 0x694A, 0x75D0, 0x7EF5, 0x8591, 
	0x8B75, 0x8FEA, 0x93DC, 0x974A, 0x9A84, 0x9D7C, 0xA053, 
	0xA30A, 0xA5CE, 0xA892, 0xAB77, 0xAE7C, 0xB195, 0xB4DC, 
	0xB851, 0xBC14, 0xC012, 0xC466, 0xC992, 0xD082, 0xD8BA, 
	0xE511, 0xF5AE, 0xFFB0, 0xFFFF	};

#endif

#if SYMBOLS_PER_BLOCK == 48
#warning SYMBOLS_PER_BLOCK 48
	// Rossi distribution 48
	static const uint16_t sumPhi[SYMBOLS_PER_BLOCK] =
	{   0x16BF, 0x29A1, 0x35FF, 0x4170, 0x49BD, 0x500B, 0x558A, 0x5AC7,
		0x5FA8, 0x63CC, 0x67BC, 0x6AFE, 0x6E74, 0x718F, 0x7457, 0x7713,
		0x7A42, 0x7D06, 0x7FE6, 0x8282, 0x8571, 0x8816, 0x8A78, 0x8D50,
		0x8FC0, 0x91FC, 0x9496, 0x972B, 0x9951, 0x9BDF, 0x9E60, 0xA0CD,
		0xA3AE, 0xA66E, 0xA995, 0xAC5C, 0xAF39, 0xB283, 0xB656, 0xBA7B,
		0xBE78, 0xC3DC, 0xCAC1, 0xD112, 0xDA1F, 0xE81B, 0xF8A9, 0xFFFF  };
#endif

#if SYMBOLS_PER_BLOCK == 64
#warning SYMBOLS_PER_BLOCK 64
	// Rossi distribution 64
	static const uint16_t sumPhi[SYMBOLS_PER_BLOCK] =
	{   0x11A1, 0x1E58, 0x2789, 0x2FA3, 0x3665, 0x3B9F, 0x403F, 0x44FA,
		0x499D, 0x4CF0, 0x509B, 0x53D4, 0x5746, 0x5AA9, 0x5DAD, 0x6060,
		0x6335, 0x6617, 0x68DD, 0x6BA8, 0x6DC9, 0x7091, 0x7301, 0x7520,
		0x77A2, 0x79FD, 0x7C1F, 0x7E7D, 0x807A, 0x82EC, 0x8589, 0x87E4,
		0x8A2F, 0x8C73, 0x8ECD, 0x90B7, 0x9300, 0x94F7, 0x96ED, 0x9961,
		0x9B7C, 0x9DA9, 0x9FEC, 0xA229, 0xA453, 0xA6E5, 0xA98B, 0xAC2A,
		0xAEE7, 0xB180, 0xB40F, 0xB740, 0xBA45, 0xBD17, 0xC04D, 0xC466,
		0xC810, 0xCC70, 0xD107, 0xD838, 0xDFA9, 0xE98B, 0xF9DA, 0xFFFF  };
#endif

	uint8_t m_buffer[DECODER_SNP_BLOCK_SIZE]; // data
	uint8_t m_bitMatrix[BIT_MATRIX_SIZE];
	uint16_t m_pckIndx = 0;
// 	uint16_t m_randDegreeCount = 0;

	/** Procedures */
	uint16_t getNumSymbols( uint16_t seed, uint16_t *s );
	void getSymbols( uint8_t *v, uint16_t n, uint16_t s );
	uint8_t testBitRC( uint16_t r, uint16_t c );
	void setBitRC( uint16_t r, uint16_t c );
	void clearBitRC( uint16_t r, uint16_t c );
	void swapRow( uint16_t r1, uint16_t r2 );
	void xorRow( uint16_t r1, uint16_t r2 );



	command uint8_t * Codec.getBuffer( uint16_t *size )
	{
		*size = /*DECODER_*/SNP_BLOCK_SIZE;
		return m_buffer;
	}

	command void Codec.startNewBlock()
	{
// 		call RandDegree.setSeed( START_DEGREE_SEED );
// 		m_randDegreeCount = 0;
		m_pckIndx = 0;
// 		for(i=0; i<BIT_MATRIX_SIZE; i++) m_bitMatrix[i]=0; 
	}

	command void Codec.encode( uint8_t *pck, uint16_t seed )
	{
		uint16_t n, i, t, s;
		uint8_t *b;
		uint8_t symbolsToMix[SYMBOLS_PER_BLOCK];

		// 1. get num of symbols to mix
		n = getNumSymbols( seed, &s );

		// 2. get symbols IDs
		getSymbols( symbolsToMix, n, s );

		// 3. build output pck
		for(i=0; i<SYMBOL_SIZE; i++) {
				pck[i] = 0x00;
		}
		for(t=0; t<SYMBOLS_PER_BLOCK; t++) {
				b = &(m_buffer[t*SYMBOL_SIZE]);
				if(symbolsToMix[t] != 0) {
					for(i=0; i<SYMBOL_SIZE; i++) {
						pck[i] ^= b[i];
					}
				}
		}
	}

	command void Codec.addToDecoder( uint8_t *pck, uint16_t seed )
	{
		uint16_t n, i, s;
		uint8_t symbolsToMix[SYMBOLS_PER_BLOCK];

		// 1. get num of symbols to mix
		n = getNumSymbols( seed, &s );

		// 2. get symbols IDs
		getSymbols( symbolsToMix, n, s );

		// 3. store data
		for(i=0; i<SYMBOL_SIZE; i++) {
				m_buffer[ m_pckIndx*SYMBOL_SIZE + i ] = pck[i];
		}

		// 4. add row to matrix
		for(i=0; i<SYMBOLS_PER_BLOCK_DIV8; i++) {
				m_bitMatrix[ m_pckIndx*SYMBOLS_PER_BLOCK_DIV8 + i ] = 0;
		}
		for(i=0; i<SYMBOLS_PER_BLOCK; i++) {
				if(symbolsToMix[i]!=0) {
					m_bitMatrix[ m_pckIndx*SYMBOLS_PER_BLOCK_DIV8 + (i>>3) ] |= (1<<(i&0x7));
				}
		}

		m_pckIndx++;
		if(m_pckIndx>=RECEIVER_SNP_BLOCK_SIZE) {  /*modified >=*/
				m_pckIndx = SYMBOLS_PER_BLOCK; /*go back at the 32th symbol*/
		}
	}

	command uint8_t Codec.decode(uint16_t pktIdx)
	{
		uint16_t cIdx, rIdx, r;
		uint16_t n_blackCol = 0;
		uint16_t find = 0;

		if (pktIdx<DECODER_SNP_BLOCK_SIZE) {
			for(r=(pktIdx*SYMBOLS_PER_BLOCK_DIV8); r<BIT_MATRIX_SIZE; r++) {
				m_bitMatrix[ r ] = 0;
			}
		}

		// all columns
		rIdx = 0;
		for(cIdx=0; cIdx<SYMBOLS_PER_BLOCK; cIdx++) {
				// look for a column with a 1
				r = rIdx;
				find = 0;
				while(!find && r<RECEIVER_SNP_BLOCK_SIZE) {
					if(testBitRC(r, cIdx)) {
						find = 1;
					} else {
						r++;
					}
				}
				if(find==0) {
					n_blackCol++;
					continue;
				}
				// move the row to the top
				if(rIdx != r) {
					swapRow(rIdx, r);
				}
				// clean the column
				for(r=0; r<RECEIVER_SNP_BLOCK_SIZE; r++) {
					if(r==rIdx) continue;
					if(testBitRC(r, cIdx)) {
						xorRow(rIdx, r);
					}
				}
				rIdx++;
		}
		if (n_blackCol==0){
			m_pckIndx = 0;
// 			for(r=0; r<BIT_MATRIX_SIZE; r++) {	
// 				m_bitMatrix[ r ] = 0;
// 			} 
		}
		return n_blackCol;
	}

	uint16_t getNumSymbols( uint16_t seed, uint16_t *s )
	{
		uint16_t i;
		uint16_t n = 1;
		n = seed;

		*s = n;

		for(i=0; i<SYMBOLS_PER_BLOCK; i++) {
				if(n<=sumPhi[i]) {
					return i+1;
				}
		}
		#ifdef DEBUG_SYNAPSE_RADIO_CODEC
		while(1) {
				call DumpToLeds.dump4bitR( 0x5, 7 );
		}
		#endif
		return 1;
	}

	void getSymbols( uint8_t *v, uint16_t n, uint16_t s )
	{
		uint16_t i, t;
		uint8_t temp[SYMBOLS_PER_BLOCK];
		// 1. clear vector IDs 
		for(i=0; i<SYMBOLS_PER_BLOCK; i++) {
				v[i] = 0;
				temp[i] = i;
		}

		// 2. set seed
		call RandPcks.setSeed( s );

		// 3. get symbols' IDs to mix
		i = n;
		// OPTIMIZATION: random permutation
		while(i != 0) {
			t = call RandPcks.getNextState( ) % (SYMBOLS_PER_BLOCK-n+i);
			v[temp[t]] = 1;
			i--;
			temp[t] = temp[SYMBOLS_PER_BLOCK-n+i];
		}

	}

	void swapRow( uint16_t r1, uint16_t r2 )
	{
		uint16_t c;
		#ifndef SWAP_IN_PLACE
		uint16_t t;
		#endif
		// swap matrix
		for(c=0; c<SYMBOLS_PER_BLOCK_DIV8; c++) {
				#ifdef SWAP_IN_PLACE
				m_bitMatrix[r1*SYMBOLS_PER_BLOCK_DIV8 + c] ^= m_bitMatrix[r2*SYMBOLS_PER_BLOCK_DIV8 + c];
				m_bitMatrix[r2*SYMBOLS_PER_BLOCK_DIV8 + c] ^= m_bitMatrix[r1*SYMBOLS_PER_BLOCK_DIV8 + c];
				m_bitMatrix[r1*SYMBOLS_PER_BLOCK_DIV8 + c] ^= m_bitMatrix[r2*SYMBOLS_PER_BLOCK_DIV8 + c];
				#else
				//r1->t   r2->r1    t->r2
				t = m_bitMatrix[r1*SYMBOLS_PER_BLOCK_DIV8 + c];
				m_bitMatrix[r1*SYMBOLS_PER_BLOCK_DIV8 + c] = m_bitMatrix[r2*SYMBOLS_PER_BLOCK_DIV8 + c];
				m_bitMatrix[r2*SYMBOLS_PER_BLOCK_DIV8 + c] = t;
				#endif
		}
		// swap data
		for(c=0; c<SYMBOL_SIZE; c++) {
				#ifdef SWAP_IN_PLACE
				m_buffer[r1*SYMBOL_SIZE + c] ^= m_buffer[r2*SYMBOL_SIZE + c];
				m_buffer[r2*SYMBOL_SIZE + c] ^= m_buffer[r1*SYMBOL_SIZE + c];
				m_buffer[r1*SYMBOL_SIZE + c] ^= m_buffer[r2*SYMBOL_SIZE + c];
				#else
				//r1->t   r2->r1    t->r2
				t = m_buffer[r1*SYMBOL_SIZE + c];
				m_buffer[r1*SYMBOL_SIZE + c] = m_buffer[r2*SYMBOL_SIZE + c];
				m_buffer[r2*SYMBOL_SIZE + c] = t;
				#endif
		}
	}

	void xorRow( uint16_t r1, uint16_t r2 )
	{
		uint16_t c;
		// xorRow matrix
		for(c=0; c<SYMBOLS_PER_BLOCK_DIV8; c++) {
				m_bitMatrix[r2*SYMBOLS_PER_BLOCK_DIV8 + c] ^= m_bitMatrix[r1*SYMBOLS_PER_BLOCK_DIV8 + c];
		}
		// xorRow data
		for(c=0; c<SYMBOL_SIZE; c++) {
				m_buffer[r2*SYMBOL_SIZE + c] ^= m_buffer[r1*SYMBOL_SIZE + c];
		}
	}

	uint8_t testBitRC( uint16_t r, uint16_t c )
	{
		return TEST_BIT( m_bitMatrix[ r*SYMBOLS_PER_BLOCK_DIV8 + (c>>3) ], c&0x7 );
	}

	void setBitRC( uint16_t r, uint16_t c )
	{
		SET_BIT( m_bitMatrix[ r*SYMBOLS_PER_BLOCK_DIV8 + (c>>3) ], c&0x7 );
	}

	void clearBitRC( uint16_t r, uint16_t c )
	{
		CLEAR_BIT(m_bitMatrix[ r*SYMBOLS_PER_BLOCK_DIV8 + (c>>3) ], c&0x7);
	}
}
