/*
 * Copyright (c) 2008 Department of Information Engineering, University of Padova, Italy
 * Contributors: Giovanni Zanca, Nicola Bui, Riccardo Crepaldi and Michele Rossi
 * All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 *       copyright notice, this list of conditions and the following
 *       disclaimer in the documentation and/or other materials provided
 *       with the distribution.
 *     * The name of the author may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#define DBG_WRONG_DATA 25
#include "Synapse.h"
#include "Synapse_msgs.h"

module Synapse_RadioP
{
	provides {
		interface Synapse_Radio;
	}

	uses {
		interface Synapse_Codec as Codec;
		interface Timer<TMilli> as Timer;
		interface Random;
		interface Synapse_Random as RandDegree;
		interface CC2420Packet;

		interface Packet as DataPacket;
		interface AMSend as DataSend;
		interface Receive as DataReceive;
		interface AMPacket as DataAMPacket;
		
#ifdef DEBUG_SYNAPSE_RADIO
		interface Packet as SerialPacket;
		interface Synapse_SerialQueue;
#endif
	}
}

#define INVALID_BLOCK_ID 0

implementation
{
	uint8_t m_blockId=INVALID_BLOCK_ID;
	uint8_t m_txBlockId=INVALID_BLOCK_ID;

	uint16_t m_currentTxSeed;
	message_t dataMsg;
	message_t *dataMsgPtr = &dataMsg;
	bool m_retry;
	bool m_receiving = FALSE;
	bool m_sending = FALSE;
	uint8_t m_pckSent=0;
	uint16_t m_pckReceived=0;
	uint8_t m_pckToSend=0;
	uint8_t m_buf[SYMBOLS_PER_BLOCK*SNP_DATA_SIZE];
	uint8_t *codecBuffer;
	uint16_t m_codecBufferSize=0;
#ifdef DEBUG_SYNAPSE_RADIO
	void SendStatMsg(uint8_t type,uint32_t a,uint32_t b,uint16_t c,uint16_t d,uint8_t e,uint8_t f);
#endif
	task void sendDataTask();

	command void Synapse_Radio.initializeSend(uint8_t blockId, uint16_t seed) {
		m_txBlockId = blockId;
		m_blockId = INVALID_BLOCK_ID;
		m_currentTxSeed = seed;
		m_receiving = FALSE;
		m_sending = TRUE;
		call RandDegree.setSeed(seed);
		m_pckSent=0;
		m_pckToSend=SYMBOLS_TO_SEND;
	}
	command void Synapse_Radio.initializeReceive(uint8_t blockId, bool retry) {
		if (m_blockId==INVALID_BLOCK_ID) retry = FALSE;
		if (!retry || (blockId!=m_blockId)) { //added   '|| blockId != m_blockId' 10-06 13:23
			call Codec.startNewBlock(); //added row 10-06 13:23
			m_blockId = blockId;
			m_pckReceived=0;
		}
	}

	command uint8_t *PASS Synapse_Radio.getBuffer( uint16_t *bufferSize ) {
		if (m_codecBufferSize==0) {
			codecBuffer = call Codec.getBuffer( &m_codecBufferSize );
		}
		*bufferSize = m_codecBufferSize;
		return codecBuffer;
	}

	command error_t Synapse_Radio.sendBlock() {
		post sendDataTask();
		return SUCCESS;
	}

	command error_t Synapse_Radio.receiveBlock() {
		if(!m_sending) m_receiving = TRUE;
		call Timer.startOneShot(TOTAL_DATA_TRANSMISSION_TIME);
		return SUCCESS;
	}

#ifdef DEBUG_SYNAPSE_RADIO
	void SendStatMsg(uint8_t type,uint32_t a,uint32_t b,uint16_t c,uint16_t d,uint8_t e,uint8_t f) {
		message_t * m_debugMsg;
		Synapse_Debug_Param_Msg * m_pDebugMsg;
		m_debugMsg = call Synapse_SerialQueue.getMsg();
		m_pDebugMsg = (Synapse_Debug_Param_Msg*)(call SerialPacket.getPayload(m_debugMsg, 0));
		m_pDebugMsg->type = type;
		m_pDebugMsg->p32_1 = a;
		m_pDebugMsg->p32_2 = b;
		m_pDebugMsg->p16_1 = c;
		m_pDebugMsg->p16_2 = d;
		m_pDebugMsg->p8_1 = e;
		m_pDebugMsg->p8_2 = f;
		call Synapse_SerialQueue.send( m_debugMsg, sizeof(Synapse_Debug_Param_Msg) );
	}
#endif

	task void sendDataTask() {
		Synapse_Data_Msg* dataPayload = (Synapse_Data_Msg*)call DataPacket.getPayload(dataMsgPtr,0);
		dataPayload->blockId = m_txBlockId;
		dataPayload->seed = call RandDegree.getNextState();
		// 	SendStatMsg(0xff,dataPayload->seed,((uint32_t *)codecBuffer)[198],0,0,0,0);
		call Codec.encode( (uint8_t*)dataPayload->data, dataPayload->seed );
		call DataSend.send(AM_BROADCAST_ADDR,dataMsgPtr,sizeof(Synapse_Data_Msg));
	}

	event void DataSend.sendDone(message_t* msg, error_t error) {
		if (error!=SUCCESS) {
			post sendDataTask();
			return;
		}
		else {
			m_pckSent++;
			if(m_pckSent==m_pckToSend) {
				m_sending = FALSE;
				signal Synapse_Radio.sendBlockDone(SUCCESS);
				m_blockId=INVALID_BLOCK_ID;
			}
			else {
				post sendDataTask();
			}
		}
	}

	event message_t* DataReceive.receive(message_t* msg, void* payload, uint8_t len) {
		// 	if (((call CC2420Packet.getMetadata(msg))->crc)==FALSE) SendStatMsg(DBG_WRONG_DATA,0,0,m_pckReceived,((Synapse_Data_Msg *)payload)->blockId,m_blockId,m_receiving);
		if((((Synapse_Data_Msg *)payload)->blockId == m_blockId) && (m_blockId!=INVALID_BLOCK_ID) && m_receiving) {
			// 		SendStatMsg(0xDD,((Synapse_Data_Msg *)payload)->seed,m_pckReceived,0,0,0,0);
			call Codec.addToDecoder( (uint8_t*)(((Synapse_Data_Msg *)payload)->data), ((Synapse_Data_Msg *)payload)->seed );
			m_pckReceived++;
		}
		return msg;
	}

	event void Timer.fired() {
		uint8_t remainingRank = SYMBOLS_PER_BLOCK;
		m_receiving=FALSE;
		if (m_pckReceived>=SYMBOLS_PER_BLOCK) {
			remainingRank = call Codec.decode(m_pckReceived);
		}
		if (remainingRank==0) {
			signal Synapse_Radio.receiveBlockDone(codecBuffer, m_codecBufferSize, m_blockId, SUCCESS);
			m_pckReceived=0;
		} else {
			signal Synapse_Radio.receiveBlockDone(codecBuffer, m_pckReceived*SNP_DATA_SIZE, m_blockId, FAIL );
		}
	}
}

