/*
 * Copyright (c) 2008 Department of Information Engineering, University of Padova, Italy
 * Contributors: Giovanni Zanca, Nicola Bui, Riccardo Crepaldi and Michele Rossi
 * All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 *       copyright notice, this list of conditions and the following
 *       disclaimer in the documentation and/or other materials provided
 *       with the distribution.
 *     * The name of the author may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <msp430x16x.h>
#include "bootloader.h"
#include "Synapse_Partition.h"
#include "asm_utils.h"
#include "internal_flash.h"
#include "external_flash.h"
#include "leds.h"

//#define DEBUG_SYNAPSE_BOOTLOADER


inline void deleteWorkingPartitions(){
/**
 *  Bootloader writes partition entries only at the end of programming action
 *  so there can't be a working partition.
 *  Nothing to do.
 */
}

void formatExtFlash() {
	externalFlash_Erase();
	// First Entry is partition table
	SynapsePartitionEntry pe;
	pe.flags = SPE_USED ^ 0xFF;
	pe.hashName = 0xABCD;
	pe.startAddr = 0x0000;
	pe.progStartOffs = 0x0000;
	pe.size = EF_PAGE_SIZE;
	externalFlash_Write(EF_PARTITION_TABLE_ADDR, (uint8_t*) (&pe),
			S_PARTITION_ENTRY_SIZE);
}

void storeInExtFlash() {
	uint8_t find = 0;
	uint32_t startAddr = 0x00;
	uint16_t offRead = 0x00;
	SynapsePartitionEntry pe;
	// Find first empty entry in partition table
	while (find == 0) {
		if (offRead > EF_PAGE_SIZE) {
			while (1) {
				blink12();
			}
		}
		externalFlash_Read(offRead, (uint8_t*) (&pe), S_PARTITION_ENTRY_SIZE);
		if ((~pe.flags) & SPE_USED) { //partition is used
			startAddr = pe.startAddr + pe.size;
			offRead += S_PARTITION_ENTRY_SIZE;
		} else {
			find = 1;
			pe.flags = (SPE_USED | SPE_PROG | SPE_WORKING) ^ 0XFF;
			pe.hashName = appID;
			pe.startAddr = startAddr;
			pe.size = imgSize + INTERRUPT_TABLE_SIZE;
			pe.progStartOffs = 0x0000;
			//externalFlash_Write(EF_PARTITION_TABLE_ADDR+offRead, (uint8_t*)(&pe), S_PARTITION_ENTRY_SIZE);
		}
	}

#ifdef DEBUG_SYNAPSE_BOOTLOADER
	blink(7);
#endif
	// Copy object code IF -> EF
	volatile uint8_t *p;
	uint16_t page;
	uint16_t offset;
	uint32_t offset32, imgSizeCount;
	p = (uint8_t*) PROG_START;
	offset32 = pe.startAddr;
	page = 0;
	while (offset32 >= EF_PAGE_SIZE) {
		page++;
		offset32 -= EF_PAGE_SIZE;
	}
	offset = offset32 & 0xFFFF;
	// Align start addr to a new page
	if (offset != 0) {
		pe.startAddr += EF_PAGE_SIZE - offset;
		page++;
		offset = 0;
	}
	pe.startAddr = (((uint32_t) page) << EF_PAGE_SIZE_LOG2);
	externalFlash_Write(EF_PARTITION_TABLE_ADDR + offRead, (uint8_t*) (&pe),
			S_PARTITION_ENTRY_SIZE);
	imgSizeCount = pe.size; //imgSize+INTERRUPT_TABLE_SIZE;
	while (imgSizeCount > 0) {
		while (offset < EF_PAGE_SIZE && imgSizeCount > 0) {
			if (offset == 0 && imgSizeCount > EF_PAGE_SIZE
					+ INTERRUPT_TABLE_SIZE) { //write 256 bytes at once if possible
				externalFlash_Write((((uint32_t) page) << EF_PAGE_SIZE_LOG2),
						(uint8_t*) p, EF_PAGE_SIZE);
				offset += EF_PAGE_SIZE;
				imgSizeCount -= EF_PAGE_SIZE;
				(uint8_t*) p += EF_PAGE_SIZE;
			} else if (imgSizeCount <= EF_PAGE_SIZE) { //last part of the last page is the irq table
				externalFlash_Write((((uint32_t) page) << EF_PAGE_SIZE_LOG2)
						+ (offset & 0xFF), (uint8_t*) p, imgSizeCount
						- INTERRUPT_TABLE_SIZE);
				offset += imgSizeCount - INTERRUPT_TABLE_SIZE;
				imgSizeCount = INTERRUPT_TABLE_SIZE;
				p = (uint8_t*) INTERRUPT_TABLE_ADDR;
				externalFlash_Write((((uint32_t) page) << EF_PAGE_SIZE_LOG2)
						+ (offset & 0xFF), (uint8_t*) p, INTERRUPT_TABLE_SIZE);
				offset = EF_PAGE_SIZE;
				imgSizeCount = 0;
			} else if (imgSizeCount > EF_PAGE_SIZE) {
				//we have the IRQ table divided between the start of the last page and the end of the previous one
				externalFlash_Write((((uint32_t) page) << EF_PAGE_SIZE_LOG2)
						+ (offset & 0xFF), (uint8_t*) p, imgSizeCount
						- INTERRUPT_TABLE_SIZE);
				offset += imgSizeCount - INTERRUPT_TABLE_SIZE;
				imgSizeCount = INTERRUPT_TABLE_SIZE;
				p = (uint8_t*) INTERRUPT_TABLE_ADDR;
				externalFlash_Write((((uint32_t) page) << EF_PAGE_SIZE_LOG2)
						+ (offset & 0xFF), (uint8_t*) p, EF_PAGE_SIZE
						- (offset));
				imgSizeCount -= EF_PAGE_SIZE - (offset);
				p += (EF_PAGE_SIZE - (offset));
				page++;
				offset = 0;
				externalFlash_Write((((uint32_t) page) << EF_PAGE_SIZE_LOG2)
						+ (offset & 0xFF), (uint8_t*) p, imgSizeCount);
				imgSizeCount = 0;
			}
		}
		page++;
		offset = 0;
	}

#ifdef DEBUG_SYNAPSE_BOOTLOADER
	blink12();
#endif
}

void loadIDFromExtFlash() {
	uint16_t page;
	SynapsePartitionEntry pe;
	uint8_t find = 0;
	uint16_t offRead = 0x00;
	// Search entry with hashName==appID in partition table
#ifdef DEBUG_SYNAPSE_BOOTLOADER
	blink(6);
#endif
	while (find == 0) {
		if (offRead > EF_PAGE_SIZE) {
			while (1) {
				blink12();
#ifdef PRESERVE_GOLDEN_IMAGE
				return; //reboot and load the Application in partition 1
#endif
			}
		}
		externalFlash_Read((EF_PARTITION_TABLE_ADDR << EF_PAGE_SIZE_LOG2)
				+ offRead, (uint8_t*) (&pe), S_PARTITION_ENTRY_SIZE);
		if ((pe.hashName != appID) || (((~pe.flags) & (SPE_WORKING))
				!= (SPE_WORKING))) {
			offRead += S_PARTITION_ENTRY_SIZE;
		} else {
			find = 1;
		}
	}
	// Copy object code EF -> IF
	uint16_t offset;
	uint32_t offset32, imgSizeCount;
	volatile uint8_t *p;
	p = (uint8_t*) PROG_START; //IntFLash program start address (i'll jump there)
	imgSizeCount = pe.size - pe.progStartOffs; //TODO: verify the progStartOffset correct functioning
#ifdef DEBUG_SYNAPSE_BOOTLOADER
	blink(6);
#endif
	internalFlash_erase((uint8_t*) p, PROG_FLASH_SIZE);
	internalFlash_erase((uint8_t*) INTERRUPT_TABLE_ADDR, INTERRUPT_TABLE_SIZE);
	uint8_t buffer[EF_PAGE_SIZE];
	uint16_t i;
	offset32 = pe.startAddr + pe.progStartOffs; //TODO: verify the progStartOffset correct functioning
	page = 0;
	while (offset32 >= EF_PAGE_SIZE) {
		page++;
		offset32 -= EF_PAGE_SIZE;
	}
	offset = offset32 & 0xFFFF;
	if (offset != 0) { //align with new page
		page++;
		offset = 0;
	}
#ifdef DEBUG_SYNAPSE_BOOTLOADER
	blink(5);
#endif
	while (imgSizeCount > 0) {
		while (offset < EF_PAGE_SIZE && imgSizeCount > 0) {
			externalFlash_Read((((uint32_t) page) << EF_PAGE_SIZE_LOG2)
					+ (offset & 0xFF), &buffer[0], EF_PAGE_SIZE);
			for (i = 0; (i < EF_PAGE_SIZE && imgSizeCount > 0); i++) {
				internalFlash_write8(p, buffer[i]);
				offset++;
				imgSizeCount--;
				if (imgSizeCount == INTERRUPT_TABLE_SIZE) {
					p = (uint8_t*) INTERRUPT_TABLE_ADDR;
				} else {
					p++;
				}
			}
		}
		page++;
		offset = 0;
	}
}

void loadPartitionFromExtFlash() //same as above but specify Part num and not ID
{
	uint16_t page;
	SynapsePartitionEntry pe;
	uint16_t offRead = 0x00;
	// Search entry in partition table
#ifdef DEBUG_SYNAPSE_BOOTLOADER
	blink(1);
#endif
	if (imgToLoad > MAX_PARTITION_ENTRY) {
		while (1) {
			blink12();
		}
	}
	offRead = imgToLoad * S_PARTITION_ENTRY_SIZE;
	externalFlash_Read(
			(EF_PARTITION_TABLE_ADDR << EF_PAGE_SIZE_LOG2) + offRead,
			(uint8_t*) (&pe), S_PARTITION_ENTRY_SIZE);
	// Copy object code EF -> IF
	uint16_t offset;
	uint32_t offset32, imgSizeCount;
	volatile uint8_t *p;
	uint8_t buffer[EF_PAGE_SIZE];
	uint16_t i;
	p = (uint8_t*) PROG_START;
	if ((!((~pe.flags) & SPE_USED)) || (!((~pe.flags) & (SPE_WORKING)))) {
		while (1) {
			blink(5);
		}
	}
	imgSizeCount = pe.size - pe.progStartOffs; //TODO: check progStartOffs
#ifdef DEBUG_SYNAPSE_BOOTLOADER
	blink(2);
#endif
	internalFlash_erase((uint8_t*) p, PROG_FLASH_SIZE);
#ifdef DEBUG_SYNAPSE_BOOTLOADER
	blink(3);
#endif
	internalFlash_erase((uint8_t*) INTERRUPT_TABLE_ADDR, INTERRUPT_TABLE_SIZE);
	offset32 = pe.startAddr + pe.progStartOffs; //TODO: check progStartOffset
	page = 0;
#ifdef DEBUG_SYNAPSE_BOOTLOADER
	blink(4);
#endif
	while (offset32 >= EF_PAGE_SIZE) {
		page++;
		offset32 -= EF_PAGE_SIZE;
	}
	offset = offset32 & 0xFFFF;
	if (offset != 0) { //align with new page
		page++;
		offset = 0;
	}
#ifdef DEBUG_SYNAPSE_BOOTLOADER
	blink(5);
#endif
	while (imgSizeCount > 0) {
		while (offset < EF_PAGE_SIZE && imgSizeCount > 0) {
			externalFlash_Read((((uint32_t) page) << EF_PAGE_SIZE_LOG2)
					+ (offset & 0xFF), &buffer[0], EF_PAGE_SIZE);
			for (i = 0; (i < EF_PAGE_SIZE && imgSizeCount > 0); i++) {
				internalFlash_write8(p, buffer[i]);
				offset++;
				imgSizeCount--;
				if (imgSizeCount == INTERRUPT_TABLE_SIZE) {
					p = (uint8_t*) INTERRUPT_TABLE_ADDR;
				} else {
					p++;
				}
			}
		}
		page++;
		offset = 0;
	}
}

int main() {
	uint16_t t[5];
	uint8_t i;
	dint(); //Interrupt disable
	nop();
	WDTCTL = WDTPW + WDTHOLD; //watchdog disable
	setPinDirection();

	// CLOCK max speed
	BCSCTL1 = RSEL0 | RSEL1 | RSEL2 | XT2OFF;
	DCOCTL = DCO0 | DCO1 | DCO2;

	flash(4);

	initializeExternalFlash();
	for (i = 0; i < 4; i++) {
		t[i] = 0;
	}
	//TODO: verify NODEID after each bootloader commands
	t[4] = ifNodeId; //copy the node ID from the ROM before overwriting it

	switch (opcode) {
	case SBC_NONE: { // 0
		// start application
		deleteWorkingPartitions();
		internalFlash_writeSegment(IF_SEGMENT_A, 0, t, 1);
		break;
	}
	case SBC_FORMAT: { // 1
		formatExtFlash();
		t[0] = SBC_ONLY_FORMAT;
		t[3] = if_sn;
		internalFlash_writeSegment(IF_SEGMENT_A, 0, t, 5);
		// format ended
		while (1) {
			blink13();
		}
		break;
	}

#ifdef PRESERVE_GOLDEN_IMAGE
	case SBC_FORMAT_SPECIAL: { // 17
		formatExtFlash();
		t[0] = SBC_STORE;
		t[1] = GOLDEN_IMAGE_ID;
		t[2] = GOLDEN_IMAGE_SIZE;
		t[3] = if_sn;
		internalFlash_writeSegment(IF_SEGMENT_A, 0, t, 5);
		// format ended - Restart bootloader and store the GOLDEN_IMAGE
		blink(7);
		__asm__ __volatile__ ("br #0x4000\n\t" ::);
		break;
	}
#endif
	case SBC_STORE: { // 2
		deleteWorkingPartitions();
		storeInExtFlash();
		t[3] = if_sn;
		internalFlash_writeSegment(IF_SEGMENT_A, 0, t, 5);
		break;
	}
	case SBC_LOAD_ID: { // 3
		deleteWorkingPartitions();
		loadIDFromExtFlash(); // and then load GOLDEN_IMG at next reboot
#ifdef PRESERVE_GOLDEN_IMAGE
		t[0] = SBC_LOAD_PARTITION;
		t[1] = 0x1;
#endif
		t[3] = if_sn;
		internalFlash_writeSegment(IF_SEGMENT_A, 0, t, 5);
		break;
	}
	case SBC_LOAD_PARTITION: { // 4
		deleteWorkingPartitions();
		loadPartitionFromExtFlash();
#ifdef PRESERVE_GOLDEN_IMAGE
		if (imgToLoad != 0x1) {
			t[0] = SBC_LOAD_PARTITION;
			t[1] = 0x1;
		}
#endif
		t[3] = if_sn;
		internalFlash_writeSegment(IF_SEGMENT_A, 0, t, 5);
		break;
	}
	case SBC_INTERNAL_FORMAT: { // 15
		formatExtFlash();
		t[0] = SBC_ONLY_FORMAT;
		t[3] = if_sn;
		internalFlash_writeSegment(IF_SEGMENT_A, 0, t, 5);
		break;
	}
	case SBC_ONLY_FORMAT: { // 16
		while (1) {
			blink(6);
		}
		break;
	}
	default: {
		// unknown opcode: clear inf mem and boot
		for (i = 0; i < 2; i++) {
			blink12();
		}
		internalFlash_writeSegment(IF_SEGMENT_A, 0, t, 5);
	}
	}

	// Start application
	blink(7);
	// 	ME1 &= ~USPIE0;  //FIXME check compatibility with TinyOS1
	// 	U0CTL &= ~SWRST;
	__asm__ __volatile__ ("br #0x5000\n\t" ::); //jump to programStart. This must be the same address specified in the makerules

	return 0;
}
