/*
 * Copyright (c) 2008 Department of Information Engineering, University of Padova, Italy
 * Contributors: Giovanni Zanca, Nicola Bui, Riccardo Crepaldi and Michele Rossi
 * All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 *       copyright notice, this list of conditions and the following
 *       disclaimer in the documentation and/or other materials provided
 *       with the distribution.
 *     * The name of the author may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "leds.h"
#include "external_flash.h"

inline void initializeExternalFlash() {
	MAKE_FLASH_CS_OUTPUT();
	MAKE_FLASH_HOLD_OUTPUT();
	SET_FLASH_HOLD_PIN();

	// 8-bit char, SPI-mode, USART as master
	U0CTL = SWRST | CHAR | SYNC | MM;
	// 3-pin + half-cycle delayed UCLK
	U0TCTL |= STC + CKPH + SSEL_SMCLK;
	// as fast as possible
	U0BR0 = 0x02;
	U0BR1 = 0;
	// enable SPI
	ME1 |= USPIE0;
	U0CTL &= ~SWRST;
	// clear interrupts
	IFG1 = 0;
	//end - from TOSBoot
}

/**
 *  SPI R/W
 */
int isTxEmpty() {
	if (U0TCTL & TXEPT) {
		return 1;
	}
	return 0;
}

int isTxIntrPending() {
	if (IFG1 & UTXIFG0) {
		IFG1 &= ~UTXIFG0;
		return 1;
	}
	return 0;
}

int isRxIntrPending() {
	if (IFG1 & URXIFG0) {
		IFG1 &= ~URXIFG0;
		return 1;
	}
	return 0;
}

uint8_t externalFlash_startCommandRW(uint8_t *cmdBuf, uint16_t n) {
	uint8_t rxData=0;
	uint8_t i;
	CLR_FLASH_SELECT_PIN();
	for (i = 0; i < n; i++) {
		U0TXBUF = cmdBuf[i];
		while (isTxIntrPending() != 1)
			;
		rxData = U0RXBUF;
	}
	return rxData;
}

void externalFlash_commandR(uint8_t *rxBuf, uint16_t n) {
	uint16_t i;
	for (i = 0; i < n; i++) {
		U0TXBUF = 0x0;
		while (!(IFG1 & URXIFG0))
			;
		rxBuf[i] = U0RXBUF;
		IFG1 &= ~URXIFG0;
	}
}

void externalFlash_commandW(uint8_t *txBuf, uint16_t n) {
	uint16_t i, rxBuf;
	for (i = 0; i < n; i++) {
		U0TXBUF = txBuf[i];
		while (!(IFG1 & URXIFG0))
			;
		rxBuf = U0RXBUF;
		IFG1 &= ~URXIFG0;
	}
}

inline void externalFlash_endCommand() {
	SET_FLASH_SELECT_PIN();
}

void externalFlash_WakeUp() {
	int8_t cmdBuf[5];
	int i;
	for (i = 0; i < 5; i++) {
		cmdBuf[i] = RES;
	}
	externalFlash_startCommandRW(cmdBuf, 5);
	externalFlash_endCommand();
	waitMicro(20000);
}

void externalFlash_Read(uint32_t offset, uint8_t *data, uint16_t n) {
	uint8_t cmdBuf[4];
	//sum base address to offset and check upper bound
	if (offset >= (uint32_t) EF_VOLUME_SIZE)
		return;
	if (n > (uint32_t)(EF_VOLUME_SIZE - offset))
		n = (EF_VOLUME_SIZE - offset);
	offset += EF_VOLUME_BASE;
	externalFlash_WakeUp();
	cmdBuf[0] = READ;
	cmdBuf[1] = (uint8_t)((offset >> 16) & 0xFF);
	cmdBuf[2] = (uint8_t)((offset >> 8) & 0xFF);
	cmdBuf[3] = (uint8_t)(offset & 0xFF);
	externalFlash_startCommandRW(cmdBuf, 4);
	externalFlash_commandR(data, n);
	externalFlash_endCommand();
	waitMicro(20000); //TODO: verify if we can do better
}

void externalFlash_Write(uint32_t offset, uint8_t *data, uint16_t n) {
	uint8_t cmdBuf[4];
	//sum base address to offset and check upper bound
	if (offset >= (uint32_t) EF_VOLUME_SIZE)
		return;
	if (n > (uint32_t)(EF_VOLUME_SIZE - offset))
		n = (EF_VOLUME_SIZE - offset);
	offset += EF_VOLUME_BASE;
	externalFlash_WakeUp();
	cmdBuf[0] = PP;
	cmdBuf[1] = (uint8_t)((offset >> 16) & 0xFF);
	cmdBuf[2] = (uint8_t)((offset >> 8) & 0xFF);
	cmdBuf[3] = (uint8_t)(offset & 0xFF);
	externalFlash_write_enable();
	externalFlash_startCommandRW(cmdBuf, 4);
	externalFlash_commandW(data, n);
	externalFlash_endCommand();
	externalFlash_waitForWip();
}

void externalFlash_Erase() {
	int8_t cmdBuf[4];
	uint32_t sec_addr;
	externalFlash_clearSR();
#ifdef DEBUG_SYNAPSE_BOOTLOADER
	blink(4);
	externalFlash_BlinkSR();
	blink(4);
#endif
	cmdBuf[0] = SE; //we erase sector by sector to avoid refuse in case some BP bits are set
	//for(sec_addr=1 ; sec_addr < 0xFFFFE ; sec_addr+=0x10000){
	//delete only target sectors
	for (sec_addr = EF_VOLUME_BASE; sec_addr < (uint32_t)(EF_VOLUME_BASE
			+ EF_VOLUME_SIZE); sec_addr += 0x10000) {
		externalFlash_write_enable();
		cmdBuf[1] = (uint8_t)((sec_addr >> 16) & 0xFF);
		cmdBuf[2] = (uint8_t)((sec_addr >> 8) & 0xFF);
		cmdBuf[3] = (uint8_t)(sec_addr & 0xFF);
		externalFlash_startCommandRW(cmdBuf, 4);
		externalFlash_endCommand();
		externalFlash_waitForWip();
	}
	waitMicro(20000); //TODO: verify if it's necessary to wait like this
}

#ifdef DEBUG_SYNAPSE_BOOTLOADER
void externalFlash_BlinkSR() {
	int8_t cmdBuf[2] ,i;
	uint8_t SR[2];
	externalFlash_WakeUp();
	externalFlash_write_enable();
	//cmdBuf[0]=WRSR;
	cmdBuf[0]=BE; //bulk erase
	cmdBuf[1] = 0x0;
	externalFlash_startCommandRW(cmdBuf, 1);
	externalFlash_endCommand();
	cmdBuf[0] = RDSR;
	externalFlash_startCommandRW(cmdBuf, 1);
	externalFlash_commandR(&SR[0],2);
	externalFlash_endCommand();
	blink(7);
	blink(0);
	for(i=0; i<8; i++) { //SR - 1st reading
		flash((SR[0]>>i & 1)+1);
		flash(0);
	}
	blink(0);
	blink(7);
	waitMicro(20000);//TODO: verify if it's necessary to wait like this
}
#endif

void externalFlash_clearSR() {
	int8_t cmdBuf[2];
	externalFlash_WakeUp();
	externalFlash_write_enable();
	cmdBuf[0] = WRSR;
	cmdBuf[1] = 0x00;
	externalFlash_startCommandRW(cmdBuf, 1);
	externalFlash_endCommand();
	externalFlash_waitForWip();
	waitMicro(20000);//TODO: verify if it's necessary to wait like this
}

void externalFlash_waitForWip() { //wait while write is in progress
	int8_t cmdBuf[1];
	uint8_t SR = 0xFF;
	externalFlash_WakeUp();
	cmdBuf[0] = RDSR;
	externalFlash_startCommandRW(cmdBuf, 1);
	while (SR & 0x1) {
		externalFlash_commandR(&SR, 1); //read from SR until wip bit become 0
		flash(1);
	}
	externalFlash_endCommand();//TODO: verify if it's necessary to wait like this
}

#ifdef DEBUG_SYNAPSE_BOOTLOADER
void externalFlash_BlinkABit() {
	int8_t cmdBuf[4] ,i;
	uint8_t rbyte;
	externalFlash_WakeUp();
	cmdBuf[0] = READ;
	cmdBuf[1] = 0;
	cmdBuf[2] = 0;
	cmdBuf[3] = 15;
	externalFlash_startCommandRW(cmdBuf, 4);
	externalFlash_commandR(&rbyte,1);
	externalFlash_endCommand();
	blink(7);
	blink(0);
	for(i=0; i<8; i++) { //byte reading
		flash((rbyte>>i & 1)+1);
		flash(0);
	}
	blink(0);
	blink(7);
	waitMicro(20000);
}
#endif

void externalFlash_write_enable() {
	int8_t cmdBuf[1];
	cmdBuf[0] = WREN;
	externalFlash_startCommandRW(cmdBuf, 1);
	externalFlash_endCommand();
	waitMicro(20000);
}

void externalFlash_write_disable() {
	int8_t cmdBuf[1];
	cmdBuf[0] = WRDI;
	externalFlash_startCommandRW(cmdBuf, 1);
	externalFlash_endCommand();
	waitMicro(20000);
}
