/*
 * Copyright (c) 2008 Department of Information Engineering, University of Padova, Italy
 * Contributors: Giovanni Zanca, Nicola Bui, Riccardo Crepaldi and Michele Rossi
 * All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 *       copyright notice, this list of conditions and the following
 *       disclaimer in the documentation and/or other materials provided
 *       with the distribution.
 *     * The name of the author may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


#include "Synapse_Partition.h"
#include "FlashManagerMessages.h"
#include "AM.h"

 
module SerialFlashManagerP{
	provides{ 
		interface Init;
		}
	uses{
		interface Boot;		
		interface Packet as SerialPacket;		
		interface SplitControl as SerialSplitControl;	
		interface Receive  as CommandReceive;
		interface AMSend  as CommandSend;
		interface Receive  as DataReceive;
		interface AMSend  as DataSend;
				
	  	interface BlockRead;
	  	interface BlockWrite;	  		  	

		interface Synapse_PartitionStorage;
		interface Synapse_BootloaderCommunication;
	  	
	  	interface Leds;

		} 
}
implementation{

	enum{
		ST_INIT=-1,
		ST_IDLE=0,
		ST_FORMATTING=1,
		ST_WRITING=2,
		ST_READING_TABLE=3,
		ST_WRITING_ENTRY=4,
		ST_READING_APPLICATION = 5,
		ST_READING=6,
		ST_SENDING_TABLE=7,		

		PARTITION_TABLE_SIZE=MAX_PARTITION_ENTRY*S_PARTITION_ENTRY_SIZE,
		};

	enum {
		FLASH_IDLE	=0,
		FLASH_MOUNTING	=1,
		FLASH_CREATING	=2,
		FLASH_READING	=3,
		FLASH_WRITING	=4,
	};
		
	uint8_t status,current_error=0;

	message_t cmd_msg;
	message_t data_msg;
	
	storage_len_t total_space;
	SynapsePartitionEntry *partition_table;
	uint8_t curEntry = 0;	
	uint16_t appID = 0x0000;	
	uint16_t curAddress = 0x0000;
	uint16_t appSize = 0x0000;
	uint16_t appOffset = 0x0000;
	uint8_t  appPos	= 0;
	uint16_t bytesToRead = 0;

	uint16_t flashStatus = FLASH_IDLE;

	void change_status(uint8_t new_status,uint8_t error);
	void wait(uint8_t t);
	void blink_error(uint8_t code);


	error_t read_table();
	error_t read_application();
	
	task void read_table_task();
	task void read_application_task();
	task void send_application_task();
	task void blinkErrorTask();
	task void send_tableEntries();
	task void send_dataChunk_task();
	task void create_partition_task();
	task void send_partition_ready_task();
	task void send_cmd_msg();

/*--------------------------------------------------
	AUX FUNCTIONS and TASKS
---------------------------------------------------*/


	void change_status(uint8_t new_status,uint8_t error){
		if(status!=new_status){
			if(new_status==ST_IDLE){
				call Leds.led2Off();
				call Leds.led1On();
			}else{
				call Leds.led2On();
				call Leds.led1Off();
			}
			status=new_status;
		}
		if(error!=0)
			call Leds.led0On();
		else
			call Leds.led0Off();
		current_error=error;
		post blinkErrorTask();
	}

	void wait(uint8_t t){
		uint16_t i,j;
		for(i=t;i>0;i--)
			for(j=0xFFFF;j>0;j--);		
	}

	void blink_error(uint8_t code){
		uint8_t old;
		uint16_t k;
		
		old=call Leds.get();
		for (k=0;k<3;k++){
			call Leds.set(0);
			wait(1);
			call Leds.set(code);
			wait(10);
			call Leds.set(0);
			wait(1);					
			call Leds.set(code>>3);
			wait(10);
			call Leds.set(0);
			wait(1);					
			call Leds.set((code>>6)&0x03);
			wait(1);					
			wait(10);
			call Leds.set(0);
			wait(5);						
		}	
		call Leds.set(old);		
	}
	
	task void blinkErrorTask(){
		if(current_error!=0)
			blink_error(current_error);			
	}	
	
	
	task void send_cmd_msg(){
		if (call CommandSend.send(AM_BROADCAST_ADDR, &cmd_msg, sizeof(message_t))!=SUCCESS){
			post send_cmd_msg();
		}
	}
	

/*---------------------------------------------------
	INTERFACE BOOT
---------------------------------------------------*/
	event void Boot.booted(){
		
	}

/*---------------------------------------------------
	INTERFACE INIT
---------------------------------------------------*/


	command error_t Init.init(){
		
		status=ST_INIT;
		
		call SerialSplitControl.start();		
		
		return SUCCESS;	
	}

/*---------------------------------------------------
	INTERFACE SERIAL SPLIT CONTROL
---------------------------------------------------*/


	event void SerialSplitControl.startDone(error_t error){
		change_status(ST_IDLE,error);
	}

	event void SerialSplitControl.stopDone(error_t error){
		
	}


/*---------------------------------------------------
	INTERFACE SYNAPSE PARTITION STORAGE
---------------------------------------------------*/


	event void Synapse_PartitionStorage.created( uint16_t id, uint16_t pos, uint16_t size, uint16_t offset, error_t error ){
		switch(status){
			case ST_WRITING_ENTRY:
				{FlashManagerCmd* payload;
				payload = (FlashManagerCmd *) call SerialPacket.getPayload(&cmd_msg,sizeof(FlashManagerCmd));
				payload->cmd=CMD_ENTRYREADY;
				payload->param16_1=id;
				payload->param16_2=size;
				payload->param8_1= (uint8_t)(pos & 0xFF);
				payload->param8_2= error;
				post send_partition_ready_task();
				}
				curAddress=0x0000;
				appID = id;
				appSize = size;
				appPos = pos;
				break;
			default:
				break;
		}
		
		
	}

	event void Synapse_PartitionStorage.mounted( uint16_t id, uint16_t pos, uint16_t size, uint16_t offset,  error_t error ){
		switch (status){
				case ST_READING_APPLICATION:
					if (error != SUCCESS){	
						change_status(ST_IDLE, 2);
						return ;
					}	
					appSize = size;
					appOffset = offset;
					curAddress = offset;
					appPos = pos;
					post send_application_task();
					break;
				case ST_WRITING_ENTRY:
//uncomment this if we don't want to allow multiple images with the same ID'
//					if (error == SUCCESS){ //entry already present and working, FAIL
//						{FlashManagerCmd* payload;
//						payload = (FlashManagerCmd *) call SerialPacket.getPayload(&cmd_msg,sizeof(FlashManagerCmd));
//						payload->cmd=CMD_ENTRYREADY;
//						payload->param16_1=id;
//						payload->param16_2=size;
//						payload->param8_1= (uint8_t)(pos & 0xFF);
//						payload->param8_2= FAIL;
//						post send_partition_ready_task();		
//						return;
//						}
//					}
					call Synapse_PartitionStorage.createPartition(appID, appSize, appOffset);
					break;
				default:
					break;
		}
	}

	event void Synapse_PartitionStorage.flagsChanged(error_t error){
		change_status(ST_IDLE,error);		
	}


/*---------------------------------------------------
	INTERFACE BLOCKREAD
---------------------------------------------------*/	
	

	event void BlockRead.computeCrcDone(storage_addr_t addr, storage_len_t len, uint16_t crc, error_t error){
		
	}

	event void BlockRead.readDone(storage_addr_t addr, void *buf, storage_len_t len, error_t error){
		switch(status){
			case ST_READING_APPLICATION:
				if (error != SUCCESS){
					change_status(ST_IDLE,7);	
				}else{
					curAddress += len;
					post send_dataChunk_task();
				}
				break;
			default:
				break;
		}
	}

	event void BlockWrite.writeDone(storage_addr_t addr, void *buf, storage_len_t len, error_t error){
		bool end = FALSE;
		FlashManagerCmd *payload;
		payload = (FlashManagerData *) call SerialPacket.getPayload(&data_msg,sizeof(FlashManagerData));
		switch (status){
			case ST_WRITING_ENTRY:
				if ((addr+len) == (appSize)){
					end = TRUE;
					call Synapse_PartitionStorage.setWorking();	
				}
				payload->cmd = CMD_ACK;
				payload->param8_1 = end;
				post send_cmd_msg();
		}
		
		
	}
	event void BlockWrite.syncDone(error_t error){
	}
	event void BlockWrite.eraseDone(error_t error){
	}


/*--------------------------------------------------
	INTERFACES COMMAND RECEIVE and SEND
---------------------------------------------------*/

	event message_t * CommandReceive.receive(message_t* msg, void* payload, uint8_t len){
		
		FlashManagerCmd *msgt;
		uint16_t hashName;
		uint16_t size;
		

		msgt = (FlashManagerCmd*)payload;
		hashName = msgt->param16_1;
		size = msgt->param16_2;
		
		switch(msgt->cmd) {
			case CMD_READTABLE:
				if (status!=ST_IDLE) return;
				post read_table_task();
				break;
			case CMD_READAPP:
				if (status!=ST_IDLE) return;
				appID = hashName;
				post read_application_task();
				break;
			case CMD_ADDENTRY:
				if (status!=ST_IDLE) return;				
				appID = hashName;
				appSize = size;
				post create_partition_task();
				break;
			case CMD_FORMAT:
				call Synapse_BootloaderCommunication.formatFlash();
				break;
			case CMD_ACK:
				break;
			default:
				break;
		}
		
		return msg;
	}

	event message_t * DataReceive.receive(message_t *msg, void *payload, uint8_t len){
		FlashManagerData *msgt;
		msgt = (FlashManagerData *) payload;
		switch(status){
			case ST_WRITING_ENTRY:
				if ((msgt->type != DATA_CHUNK)){
					change_status(ST_IDLE,15);
					return msg;
				}	
				if ((curAddress != msgt->address)){
					change_status(ST_IDLE,16);
					return msg;	
				}
				if ((curAddress + msgt->size) > appSize){
					change_status(ST_IDLE,17);
					return msg;
				}
				if (call BlockWrite.write(curAddress, msgt->data, msgt->size)!= SUCCESS){
					change_status(ST_IDLE,18);
					return msg;
				}
				curAddress+=(uint16_t)msgt->size;
				break;
			default:
				break;
		}
		return msg;
	}

	event void DataSend.sendDone(message_t* msg, error_t error){
		FlashManagerData *payload;
		payload = (FlashManagerData *) call SerialPacket.getPayload(&data_msg,sizeof(FlashManagerData));
		switch (status) {
			case ST_SENDING_TABLE:
				if (error != SUCCESS) curEntry-=(DATASIZE/S_PARTITION_ENTRY_SIZE);
				if (payload->address ==0){
					post send_tableEntries();
				}else{
					change_status(ST_IDLE,0);
				}
				break;
			case ST_READING_APPLICATION:
				if (error != SUCCESS) curAddress-=bytesToRead;
				if (payload->address ==0){
					post send_application_task(); 
				}else{
					change_status(ST_IDLE,0);
				}
				break;
			default:
				break;
		}		

	}

	event void CommandSend.sendDone(message_t* msg, error_t error){
		switch (status){
			case ST_WRITING_ENTRY:
				{FlashManagerCmd* payload;
				payload = (FlashManagerCmd *) call SerialPacket.getPayload(&cmd_msg,sizeof(FlashManagerCmd));
				if (payload->param8_2 != SUCCESS){
					change_status(ST_IDLE,4);
					return;
				}
				}
			break;			
		}
	}



/*--------------------------------------------------
	FLASH FUNCTIONS
--------------------------------------------------*/


// READ PARTITION TABLE

	task void read_table_task(){
		if (read_table() != SUCCESS) {
			post read_table_task();
		}
	}	

	error_t read_table(){
		error_t err;
		if (status!=ST_IDLE){
			return EBUSY;
		}		
		change_status(ST_READING_TABLE,0);
		curEntry = 0;
		total_space = call BlockRead.getSize();
		total_space -=PARTITION_TABLE_SIZE;
		partition_table = call Synapse_PartitionStorage.getPartitionTable();	
		change_status(ST_SENDING_TABLE,0);
		post send_tableEntries();
		return SUCCESS;			
	}

	task void send_tableEntries(){
		FlashManagerData *payload;
		error_t err;
		payload = (FlashManagerData *) call SerialPacket.getPayload(&data_msg,sizeof(FlashManagerData));
		payload->type = TABLE_ENTRY;
		payload->address = 0xFFFF;
		memcpy(payload->data, partition_table+curEntry, S_PARTITION_ENTRY_SIZE * (DATASIZE / S_PARTITION_ENTRY_SIZE));
		curEntry+=(DATASIZE/S_PARTITION_ENTRY_SIZE); 
		if ((!partitionUsed(partition_table+curEntry)) || (curEntry>=MAX_PARTITION_ENTRY))
			payload->address=1; //mark this as the last packet
		else
			payload->address=0; //to be continued
		err=call DataSend.send(AM_BROADCAST_ADDR, &data_msg, sizeof(message_t));			
		if (err  != SUCCESS){
			curEntry-=(DATASIZE/S_PARTITION_ENTRY_SIZE);
			post send_tableEntries();			
		}
	}

//READ APPLICATION ID

	task void read_application_task(){
		if (read_application()!=SUCCESS){
			post read_application_task();	
		}
		
	}

	error_t read_application(){
		error_t err;
		if (status!=ST_IDLE){
			return EBUSY;
		}		
		change_status(ST_READING_APPLICATION,0);
		err = call Synapse_PartitionStorage.mountId(appID);
		if (err != SUCCESS){
			change_status(ST_IDLE, 15);
			return err;
		}
		return SUCCESS;
	}	
	
	task void send_application_task(){
		FlashManagerData* payload; 			
		
		payload = (FlashManagerData *) call SerialPacket.getPayload(&data_msg,sizeof(FlashManagerData));
		if (status!=ST_READING_APPLICATION){
			change_status(ST_IDLE, 6);
			return;
		}
		if ((curAddress + DATASIZE) > appSize ){
			bytesToRead = appSize - curAddress;
			payload->address = 1;	
		}else{
			bytesToRead = DATASIZE;
			payload->address = 0;	
		}
		payload->size = bytesToRead;
		payload->type = DATA_CHUNK;
		if (call BlockRead.read(curAddress,payload->data,bytesToRead)!= SUCCESS){
			post send_application_task();
			return;
		}
	}
	
	task void send_dataChunk_task(){
		error_t err;
		err=call DataSend.send(AM_BROADCAST_ADDR, &data_msg, sizeof(message_t));			
		if (err  != SUCCESS){
			post send_dataChunk_task(); //add a Timeout for lost messages;			
		}		
	}
	
//	WRITE APPLICATION
	
	task void create_partition_task(){
		//1) verify if already exists
		//2) create entry
		//3) write application
		//4) set working if everything is ok
		change_status(ST_WRITING_ENTRY,0);
		call Synapse_PartitionStorage.mountId(appID);
	}
	
	task void send_partition_ready_task(){
		call CommandSend.send(AM_BROADCAST_ADDR, &cmd_msg, sizeof(message_t));
	}

}
