/*
 * Copyright (c) 2008 Department of Information Engineering, University of Padova, Italy
 * Contributors: Giovanni Zanca, Nicola Bui, Riccardo Crepaldi and Michele Rossi
 * All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 *       copyright notice, this list of conditions and the following
 *       disclaimer in the documentation and/or other materials provided
 *       with the distribution.
 *     * The name of the author may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
 
#include "Storage.h"
#include "Synapse_Partition.h"
#include "external_flash.h"
#include "Synapse.h"
//#include "printf.h"
#define INVALID_PARTITION_ID 0

/** MACROS */


module Synapse_PartitionStorageP {
	provides {
		interface Synapse_PartitionStorage[uint8_t partId];
		interface BlockRead[uint8_t partId];
		interface BlockWrite[uint8_t partId];
		command uint16_t get_NumOfImg();
	}
	uses {
		interface Boot;
		interface BlockRead as FlashRead;
		interface BlockWrite as FlashWrite;
		interface Leds;
	}
}

implementation 
{
	/** Procedures */
	void addOffset(uint16_t _offset);
	/** Tasks */
	task void signalMountedTask();
	task void signalCreatedTask();
	task void signalReadDoneTask();
	task void signalWriteDoneTask();
	task void signalSyncDoneTask();
	task void readNextPage();
	task void writeNextPage();
	task void syncAllTask();
	task void scanPartitionTableTask();
	void request(uint8_t partId);

	enum {
		NUM_STATE = uniqueCount(UQ_SYNAPSE_PARTITION_STORAGE)
	};

	enum {
		UNMOUNTED = 0,
		MOUNTING  = 1,
		CREATING  = 2,
		MOUNTED   = 3,
		FORMATTING = 4,
		SCANNING = 5,
	};

	enum {
		ACTION_NOTHING  = 0,
		ACTION_READ     = 1,
		ACTION_WRITE    = 2,
		ACTION_SYNC     = 3,
		ACTION_VALIDATE = 4,
	};

	struct StateStruct {
		// Partition entry
		uint16_t pos;
		uint16_t id;
		uint32_t size;
		uint32_t lastEndAddr;
		uint16_t peOffset;
		// R/W pointers
		uint32_t page; 		//we actually use 32 bits to keep room for direct addressing the memory
		uint32_t offset;
		storage_len_t len; 	//size to rw it is not modified
		uint16_t progStartOffs;
		uint8_t *data;
		// partition entry
		SynapsePartitionEntry pe;
		// state var
		uint8_t mounted;
		uint8_t action;
		uint32_t addr; //Base address for current action
	} states[NUM_STATE];

	norace uint8_t clientId;
	norace error_t m_errorToSignal = SUCCESS;
	norace uint16_t m_rwLen;      // actual len to rw from chip [0...EF_PAGE_SIZE]
	norace storage_len_t m_rwGlobalLen;   // global len [0...state[clientId].len]
	norace bool to_start=TRUE;
//	SynapsePartitionEntry partitionList[EF_PAGE_SIZE/sizeof(struct SynapsePartitionEntry_struct)];
	SynapsePartitionEntry partitionList[MAX_PARTITION_ENTRY];

	uint16_t numOfImg;


	task void scanPartitionTableTask(){
		states[clientId].pos = 0;
		states[clientId].id = 0;
		states[clientId].peOffset = 0;
		states[clientId].size = 0;
		states[clientId].page = 0;
		states[clientId].len = 0;
		states[clientId].offset = 0;
		states[clientId].addr = 0;
		request(clientId);
	}

	event void Boot.booted()
	{
		uint16_t i,j;
		atomic{
		if (to_start){
			to_start=FALSE;
			clientId=0;
			for(i=0; i<NUM_STATE; i++) {
				states[i].action = ACTION_NOTHING;
				states[i].mounted = UNMOUNTED;
			}
			states[clientId].mounted=SCANNING;
			post scanPartitionTableTask();
		} 
		}
	}

	command error_t Synapse_PartitionStorage.setWorking[uint8_t partId](){
		SynapsePartitionEntry *p = &(states[partId].pe);
		if (states[partId].mounted != MOUNTED) {
			return FAIL;
		}
		states[partId].action=ACTION_VALIDATE;
 		if(((p->flags)&(~(SPE_WORKING))) != (p->flags)){ 		//won't do it twice
			p->flags=(p->flags)&(~(SPE_WORKING));
			call FlashWrite.write( states[partId].peOffset, (void *)&(states[partId].pe), sizeof(struct SynapsePartitionEntry_struct));
			partitionList[states[partId].pos].flags=p->flags;
// 			partitionList[(states[partId].peOffset)/sizeof(struct SynapsePartitionEntry_struct)].flags=p->flags;
			numOfImg++;
			return (error_t)(states[partId].pos);  //FIXME remove me
		}
		else{
			states[clientId].action=ACTION_NOTHING;
			signal Synapse_PartitionStorage.flagsChanged[clientId](SUCCESS);
		}
		return SUCCESS;
		}

	command error_t Synapse_PartitionStorage.mountId[uint8_t partId]( uint16_t id )
	{
		// check for pending action
		if(states[partId].action != ACTION_NOTHING) {  
			return EBUSY;
		}
		states[partId].pos = EF_PAGE_SIZE/sizeof(struct SynapsePartitionEntry_struct);
		states[partId].id = id;
		states[partId].size = 0;
		states[partId].peOffset = states[partId].pos*sizeof(struct SynapsePartitionEntry_struct);
		states[partId].page = 0;
		states[partId].offset = 0;
		states[partId].len = 0;
		states[partId].mounted = MOUNTING;
		states[partId].action = ACTION_NOTHING;
		request(partId);
		return SUCCESS;
	}

	command error_t Synapse_PartitionStorage.createPartition[uint8_t partId]( uint16_t id,uint16_t size , uint16_t offset)
	{
		// check for pending action
		if(states[partId].action != ACTION_NOTHING) {
			return EBUSY;
		}
		states[partId].pos = 0;
		states[partId].id = id;
		states[partId].peOffset = 0;
		states[partId].size = size;
		states[partId].progStartOffs = offset;
		states[partId].mounted = CREATING;
		states[partId].action = ACTION_NOTHING;
		request(partId);
		return SUCCESS;
	}

	command SynapsePartitionEntry* Synapse_PartitionStorage.getPartitionTable[uint8_t partId](){
		return partitionList;
	}


	/** Resource */
	void request(uint8_t partId)
	{
		clientId = partId;
		//if (clientId != 0) {while(1) call Leds.set(7);}
		switch(states[clientId].mounted) {
			case MOUNTING:
			case SCANNING:
			case CREATING: {
				call FlashRead.read(states[clientId].peOffset, (void *)&(states[clientId].pe), sizeof(SynapsePartitionEntry));
				break;
			}
			case MOUNTED: {
				break;
			}
			default: {
				return;
			}
		}
		switch(states[clientId].action) {
			case ACTION_READ: {
				m_rwGlobalLen = 0;
				m_rwLen =  MIN((uint16_t)((states[partId].len)-m_rwGlobalLen),(uint16_t)( EF_PAGE_SIZE-states[partId].offset));
				if(call FlashRead.read((states[partId].page << EF_PAGE_SIZE_LOG2) | states[partId].offset, states[partId].data, m_rwLen) != SUCCESS){
					m_errorToSignal = FAIL;
					post signalReadDoneTask();
				}
				break;
			}
			case ACTION_WRITE: {
				m_rwGlobalLen = 0;
				m_rwLen = MIN((uint16_t)((states[partId].len)-m_rwGlobalLen),(uint16_t)(EF_PAGE_SIZE-states[partId].offset));
				if (call FlashWrite.write((states[partId].page << EF_PAGE_SIZE_LOG2 )| states[partId].offset, (uint8_t *)&(states[partId].data[m_rwGlobalLen]), m_rwLen) != SUCCESS){
					m_errorToSignal = FAIL;
					post signalWriteDoneTask();
				}
				break;
			}
			case ACTION_SYNC: {
				post syncAllTask();
				break;
			}
			case ACTION_NOTHING:
			default: {
				return;
			}
		}
	}

	/** BlockRead */
	command error_t BlockRead.read[uint8_t partId](storage_addr_t addr, void* buf, storage_len_t len)
	{
		// check if partition can be used
		switch(states[partId].mounted) {
			case UNMOUNTED: {
				return EOFF;
			}
			case CREATING:
			case MOUNTING: {
				return EBUSY;
			}
		}
		// check for pending action
		if(states[partId].action != ACTION_NOTHING) {
			return EBUSY;
		}
		states[partId].len = len;
		states[partId].data = buf;
		states[partId].addr = addr;
		states[partId].action = ACTION_READ;
		if(addr!=0xFFFFFFFF) {
			uint32_t offset32;
//			offset32 = states[clientId].pe.startAddr + addr-1;
			offset32 = states[clientId].pe.startAddr + addr;
			states[clientId].page = 0;
			while(offset32>=EF_PAGE_SIZE) {
				states[clientId].page++;
				offset32 -= EF_PAGE_SIZE;
			}
			states[clientId].offset = offset32 & 0xFFFF;
		}
		request(partId);
		return SUCCESS;
	}

	command error_t BlockRead.computeCrc[uint8_t partId](storage_addr_t addr, storage_len_t len, uint16_t crc)
	{
		// TODO
		return SUCCESS;
	}

	command storage_len_t BlockRead.getSize[uint8_t partId]()
	{
		return states[partId].size;
	}

	/** BlockWrite */
	command error_t BlockWrite.write[uint8_t partId](storage_addr_t addr, void* buf, storage_len_t len)
	{
		// check if partition can be used
		switch(states[partId].mounted) {
			case UNMOUNTED: {
				return EOFF;
			}
			case CREATING:
			case MOUNTING: {
				return EBUSY;
			}
		}
		// check for pending action
		if(states[partId].action != ACTION_NOTHING) {
			return EBUSY;
		}
		if((addr+len) > states[partId].size+1) return FAIL;
		states[partId].len = len;
		states[partId].data = buf;
		states[partId].addr = addr;
		states[partId].action = ACTION_WRITE;
		if(addr!=0xFFFFFFFF) {
			uint32_t offset32;
//			offset32 = states[clientId].pe.startAddr + addr-1;
			offset32 = states[clientId].pe.startAddr + addr;
			states[clientId].page = 0;
			while(offset32>=EF_PAGE_SIZE) {
				states[clientId].page++;
				offset32 -= EF_PAGE_SIZE;
			}
			states[clientId].offset = offset32 & 0xFFFF;
		}
		request(partId);
		return SUCCESS;
	}

	command error_t BlockWrite.erase[uint8_t partId]()
	{
		call FlashWrite.erase();
		return SUCCESS;
	}

	command error_t BlockWrite.sync[uint8_t partId]()
	{
		// check if partition can be used
		switch(states[partId].mounted) {
			case UNMOUNTED: {
				return EOFF;
			}
			case CREATING:
			case MOUNTING: {
				return EBUSY;
			}
		}
		// check for pending action
		if(states[partId].action != ACTION_NOTHING) {
			return EBUSY;
		}
		post syncAllTask();
		return SUCCESS;
	}

	command SynapsePartitionEntry* Synapse_PartitionStorage.get_PartitionEntry[uint8_t partId](uint16_t part_num){
		if (part_num <= (EF_PAGE_SIZE/sizeof(SynapsePartitionEntry))) return &partitionList[part_num];
		return (SynapsePartitionEntry *)NULL;
	}

	command uint16_t get_NumOfImg() {
		return numOfImg;
	}

event void FlashWrite.writeDone(storage_addr_t addr, void* buf, storage_len_t len, error_t error){
	switch(states[clientId].mounted) {
		case FORMATTING: {
			states[clientId].mounted = CREATING;
			states[clientId].peOffset += sizeof(struct SynapsePartitionEntry_struct);
			states[clientId].pos++;
			states[clientId].lastEndAddr = EF_PAGE_SIZE;
			call FlashRead.read( states[clientId].peOffset, (uint8_t *)&(states[clientId].pe), sizeof(struct SynapsePartitionEntry_struct) );
		break;
		}
		case CREATING: {
			if(error == SUCCESS) {
				m_errorToSignal=SUCCESS;
				post signalCreatedTask();
			} else {
				call FlashWrite.write(states[clientId].peOffset, (uint8_t *)&(states[clientId].pe), sizeof(struct SynapsePartitionEntry_struct) );
			}
			break;
		}
		case MOUNTED: {
			break;
		}
		default: {
		}
	}
		switch(states[clientId].action) {
			case ACTION_WRITE: {
				if(error==SUCCESS) {
					addOffset(m_rwLen);
					m_rwGlobalLen += m_rwLen;
					m_rwLen = MIN((uint16_t)((states[clientId].len)-m_rwGlobalLen), (uint16_t)(EF_PAGE_SIZE-states[clientId].offset));
					if(m_rwLen==0) {
						m_errorToSignal = error;
						post signalWriteDoneTask();
						return;
					}
				} else {
					m_errorToSignal = error;
					post signalWriteDoneTask();
					return;
				}
				post writeNextPage();
				return;
			break;
			}
			case ACTION_VALIDATE: {
				states[clientId].action=ACTION_NOTHING;
				signal Synapse_PartitionStorage.flagsChanged[clientId](error);
			break;
			}
			default: {
				return;
			}
		}
	}

	event void FlashRead.readDone(storage_addr_t addr, void* buf, storage_len_t len, error_t error)
	{
		int j,i;
		SynapsePartitionEntry *p = &(states[clientId].pe);
		switch(states[clientId].mounted) {
			case MOUNTING: {
				call Leds.led2Toggle();
				//printf("pid: %04x lf: %04x\n",p->hashName,states[clientId].id );
				if((p->hashName == states[clientId].id) && ((~(p->flags)&(SPE_WORKING))==SPE_WORKING)) {
					uint32_t offset32;
					states[clientId].size = p->size;
					states[clientId].progStartOffs = p->progStartOffs;
					offset32 = states[clientId].pe.startAddr;
					states[clientId].page = 0;
					while(offset32>=EF_PAGE_SIZE) {
						states[clientId].page++;
						offset32 -= EF_PAGE_SIZE;
					}
					states[clientId].offset = offset32 & 0xFFFF;
					m_errorToSignal = error;
					post signalMountedTask();
				} else {
					states[clientId].peOffset -= sizeof(SynapsePartitionEntry);
					states[clientId].pos--;
					if( states[clientId].pos == 0) { //entry not found
						states[clientId].mounted = UNMOUNTED;
						states[clientId].action = ACTION_NOTHING;
						signal Synapse_PartitionStorage.mounted[clientId](states[clientId].id, 0, 0,states[clientId].pe.progStartOffs, FAIL);
					}
					else{
						call FlashRead.read(states[clientId].peOffset, (uint8_t *)&(states[clientId].pe), sizeof(SynapsePartitionEntry));
					}
				}
				break;
			}
			case CREATING: {
				if((~(p->flags)&(SPE_USED))!= (SPE_USED)){
					if (states[clientId].peOffset == 0){ 	//first entry empty - fill first entry
						states[clientId].mounted = FORMATTING;
						p->flags = ~(SPE_USED | SPE_PROG | SPE_WORKING);
						p->hashName = INVALID_PARTITION_ID;
						p->startAddr = 0;
						p->size = EF_PAGE_SIZE;
						p->progStartOffs = 0x0000;
						call FlashWrite.write( states[clientId].peOffset, (void *)&(states[clientId].pe), 
													sizeof(SynapsePartitionEntry) );
						return;
					}
					else{
					// First free enty found
					// Create partition entry
					uint32_t offset32;
					p->flags = ~(SPE_USED | SPE_PROG);
					p->hashName = states[clientId].id;
					p->startAddr = states[clientId].lastEndAddr;
					offset32 = states[clientId].pe.startAddr;
					states[clientId].page = 0;
					while(offset32>=EF_PAGE_SIZE) {
						states[clientId].page++;
						offset32 -= EF_PAGE_SIZE;
					}
					states[clientId].offset = offset32 & 0xFFFF;
					if(states[clientId].offset!=0) {
						p->startAddr += EF_PAGE_SIZE-states[clientId].offset;
						states[clientId].lastEndAddr += EF_PAGE_SIZE-states[clientId].offset;
						states[clientId].page++;
						states[clientId].offset = 0;
					}
					if((states[clientId].size+((states[clientId].page<<EF_PAGE_SIZE_LOG2)|states[clientId].offset)) > call FlashRead.getSize()){
						m_errorToSignal = FAIL;
						post signalCreatedTask();
						return;
					}
					p->size = states[clientId].size;
					p->progStartOffs = states[clientId].progStartOffs;
					// Write partition entry
					call FlashWrite.write(states[clientId].peOffset, (void *)&(states[clientId].pe), sizeof(struct SynapsePartitionEntry_struct));
					}
				} 
				else {
					// Read next entry
					states[clientId].peOffset += sizeof(struct SynapsePartitionEntry_struct);
					states[clientId].pos++;
					if(states[clientId].peOffset >= EF_PAGE_SIZE) {
						//ERR: no free entry found
						states[clientId].mounted = UNMOUNTED;
						states[clientId].action = ACTION_NOTHING;
						signal Synapse_PartitionStorage.created[clientId](states[clientId].id, 0, 0,0, FAIL);
					}
					states[clientId].lastEndAddr = p->startAddr + p->size;
					call FlashRead.read( states[clientId].peOffset, (uint8_t *)&(states[clientId].pe), sizeof(struct SynapsePartitionEntry_struct) );
				}
				break;
			}
			case MOUNTED: {
				break;
			}
			case SCANNING:{
				if((~(p->flags)&(SPE_USED))==SPE_USED) numOfImg++;
				partitionList[states[clientId].pos]=states[clientId].pe;
				states[clientId].peOffset += sizeof(struct SynapsePartitionEntry_struct);
				states[clientId].pos++;
				if(  (states[clientId].peOffset >= EF_PAGE_SIZE) || (states[clientId].pos >= MAX_PARTITION_ENTRY)) { //scan ended
					states[clientId].peOffset=0;
					states[clientId].mounted = UNMOUNTED;
					states[clientId].action = ACTION_NOTHING;
				}
				else{
					call FlashRead.read(states[clientId].peOffset, (uint8_t *)&(states[clientId].pe), sizeof(SynapsePartitionEntry));
				}
			break;
			}
			default: {
				return;
			}
		}
		switch(states[clientId].action) {
			case ACTION_READ: {
				if(error==SUCCESS) {
					addOffset(m_rwLen);
					m_rwGlobalLen += m_rwLen;
					m_rwLen = MIN((uint16_t)((states[clientId].len)-m_rwGlobalLen), (uint16_t)(EF_PAGE_SIZE-states[clientId].offset));
					if(m_rwGlobalLen == states[clientId].len) {
						m_errorToSignal = SUCCESS;
						post signalReadDoneTask();
						return;
					}
				} else {
					m_errorToSignal = FAIL;
					post signalReadDoneTask();
					return;
				}
				post readNextPage();
				break;
			}
			case ACTION_WRITE: {
				break;
			}
			case ACTION_NOTHING:
			default: {
				return;
			}
		}
	}

	event void FlashWrite.eraseDone(error_t error ){}
	event void FlashRead.computeCrcDone(storage_addr_t addr, storage_len_t len,
				uint16_t crc, error_t error){}


	/** PROCEDURES */
	void addOffset(uint16_t _offset)
	{
		atomic {
			states[clientId].offset += _offset;
			while(states[clientId].offset>=EF_PAGE_SIZE) {
				states[clientId].offset -= EF_PAGE_SIZE;
				states[clientId].page++;
			}
		}
	}

	/** TASKS */
	task void signalMountedTask()
	{
		states[clientId].mounted = MOUNTED;
		states[clientId].action = ACTION_NOTHING;
		signal Synapse_PartitionStorage.mounted[clientId](states[clientId].id, states[clientId].pos, (uint16_t)states[clientId].size,(states[clientId].pe).progStartOffs, m_errorToSignal);
	}
 
	task void signalCreatedTask()
	{
// 		call MemoryWatchDog(&(states[clientId].pos));
		if(m_errorToSignal == SUCCESS){
			partitionList[states[clientId].pos]=states[clientId].pe;
			states[clientId].mounted = MOUNTED;
		}
		else states[clientId].mounted = UNMOUNTED;
		states[clientId].action = ACTION_NOTHING;
		signal Synapse_PartitionStorage.created[clientId](states[clientId].id, states[clientId].pos, (uint16_t)states[clientId].size,states[clientId].progStartOffs, m_errorToSignal);
	}

	task void signalReadDoneTask()
	{
		states[clientId].action = ACTION_NOTHING;
		signal BlockRead.readDone[clientId](states[clientId].addr, states[clientId].data, states[clientId].len, m_errorToSignal);
	}

	task void signalWriteDoneTask()
	{
		states[clientId].action = ACTION_NOTHING;
		signal BlockWrite.writeDone[clientId](states[clientId].addr, states[clientId].data, states[clientId].len, m_errorToSignal);
	}

	task void signalSyncDoneTask()
	{
		signal BlockWrite.syncDone[clientId]( m_errorToSignal );
	}

	task void readNextPage()
	{
		call FlashRead.read((states[clientId].page <<  EF_PAGE_SIZE_LOG2) | states[clientId].offset, (uint8_t *)&(states[clientId].data[m_rwGlobalLen]), m_rwLen);
	}

	task void writeNextPage()
	{
		call FlashWrite.write((states[clientId].page <<  EF_PAGE_SIZE_LOG2) |  states[clientId].offset, (uint8_t *)&(states[clientId].data[m_rwGlobalLen]), m_rwLen);
	}

	task void syncAllTask()
	{
		m_errorToSignal = SUCCESS;
		call FlashWrite.sync();
	}


event void FlashWrite.syncDone(error_t error){
	post signalSyncDoneTask();
	}

	/**
	*  Defaults
	*************************************************************************/
	default event void Synapse_PartitionStorage.mounted[uint8_t partId]( uint16_t id, uint16_t pos, uint16_t size, uint16_t offset, error_t error )
	{ }
	default event void Synapse_PartitionStorage.created[uint8_t partId]( uint16_t id, uint16_t pos, uint16_t size, uint16_t offset, error_t error )
	{ }
	default event void Synapse_PartitionStorage.flagsChanged[uint8_t partId](error_t error )
	{ }
	default event void BlockRead.readDone[uint8_t partId](storage_addr_t addr, void* buf, storage_len_t len, error_t error)
	{ }
	default event void BlockRead.computeCrcDone[uint8_t partId](storage_addr_t addr, storage_len_t len, uint16_t crc, error_t error)
	{ }
	default event void BlockWrite.writeDone[uint8_t partId](storage_addr_t addr, void* buf, storage_len_t len, error_t error)
	{ }
	default event void BlockWrite.eraseDone[uint8_t partId](error_t error){
		signal BlockWrite.eraseDone[partId](SUCCESS); 
	}
	default event void BlockWrite.syncDone[uint8_t partId](error_t error)
	{ }

}
