/*
 * Copyright (c) 2008 Department of Information Engineering, University of Padova, Italy
 * Contributors: Giovanni Zanca, Nicola Bui, Riccardo Crepaldi and Michele Rossi
 * All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 *       copyright notice, this list of conditions and the following
 *       disclaimer in the documentation and/or other materials provided
 *       with the distribution.
 *     * The name of the author may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#define SERIAL_BUFFER_SIZE  30

module Synapse_SerialQueueP
{
	provides {
		interface Synapse_SerialQueue;
	}
	uses {
		interface Boot;
		interface SplitControl as SerialControl;
		interface AMSend as SerialSend;
		interface AMPacket as SerialAMPacket;
	}
}

implementation
{
	enum{
		STATE_FREE  =   0,
		STATE_LOCK  =   1,
		STATE_SEND  =   2
	};
	message_t m_buffer[SERIAL_BUFFER_SIZE];
	uint16_t m_state[SERIAL_BUFFER_SIZE];
	uint16_t m_msgSize[SERIAL_BUFFER_SIZE];
	uint16_t m_size;
	uint16_t m_sending;

	task void sendTask();

	event void Boot.booted() {
		uint16_t i;
		m_size = 0;
		m_sending = 0;
		for(i=0; i<SERIAL_BUFFER_SIZE; i++) {
			m_state[i] = STATE_FREE;
			m_msgSize[i] = 0;
		}
		call SerialControl.start();
	}

	event void SerialControl.startDone(error_t err) {
		if(err!=SUCCESS) {
			call SerialControl.start();
			return;
		}
	}

	event void SerialControl.stopDone(error_t err) {
	}

	command message_t * Synapse_SerialQueue.getMsg( ) {
		uint16_t i, find = 0;
		for(i=0; i<SERIAL_BUFFER_SIZE; i++) {
			if(m_state[i] == STATE_FREE) {
				find = 1;
				break;
			}
		}
		if(find==1) {
			m_state[i] = STATE_LOCK;
			return &(m_buffer[i]);
		} else {
			return 0;
		}
	}

	command  error_t Synapse_SerialQueue.send( message_t * msg, uint16_t size )
	{
		uint16_t i, find = 0;
		for(i=0; i<SERIAL_BUFFER_SIZE; i++) {
			if(&(m_buffer[i]) == msg) {
				find = 1;
				break;
			}
		}
		if(find==1) {
			if(m_state[i]==STATE_LOCK) {
				m_state[i] = STATE_SEND;
				m_msgSize[i] = size;
				m_size++;
				if(m_sending==0) {
					post sendTask();
				}
				return SUCCESS;
			} else {
				return FAIL;
			}
		} else {
			return FAIL;
		}
	}

	command  error_t Synapse_SerialQueue.isEmpty()
	{
		if(m_size>0) {
			return FAIL;
		}
		return SUCCESS;
	}

	event void SerialSend.sendDone(message_t* msg, error_t error)
	{
		uint16_t i, find = 0;
		if(error!=SUCCESS) {
			post sendTask();
			return;
		}
		m_sending = 0;
		for(i=0; i<SERIAL_BUFFER_SIZE; i++) {
			if(&(m_buffer[i]) == msg) {
				find = 1;
				break;
			}
		}
		if(find==1) {
			if(m_state[i]==STATE_SEND) {
				m_state[i] = STATE_FREE;
				m_size--;
			} else {
				m_size--;
			}
		}
		if(m_size>0) {
			post sendTask();
		}
	}

	task void sendTask()
	{
		uint16_t i, find = 0;
		for(i=0; i<SERIAL_BUFFER_SIZE; i++) {
			if(m_state[i] == STATE_SEND) {
				find = 1;
				break;
			}
		}
		if(find==1) {
			m_sending = 1;
			call SerialAMPacket.setSource(&m_buffer[i], TOS_NODE_ID);
			call SerialSend.send(AM_BROADCAST_ADDR, &m_buffer[i], m_msgSize[i]);
		}
	}
}
